/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-11-12
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <math.h>

#include "AbstractFKFunction.h"
#include "FKGradientSearch.h"
#include "FKGridSearch.h"
#include "FKSteeringOneComponent.h"
#include "FKCrossSpectrum.h"

#include "ConvFKVertical.h"
#include "ConvFKRadial.h"
#include "ConvFKTransverse.h"
#include "ConvFKRayleigh.h"
#include "OmniFKVertical.h"
#include "OmniFKRayleigh.h"
#include "PoggiFKVertical.h"
#include "PoggiFKRadial.h"
#include "HRFKVertical.h"
#include "HRFKRadial.h"
#include "HRFKTransverse.h"
#include "HRFKRayleigh.h"
#include "HRFKRayleighRadial.h"
#include "HRFKRayleighFixedEll.h"
#include "HRFKDirectRayleigh.h"
#include "HRFKDirectRayleighFixedEll.h"
#include "HRFKDirectRayleighRadial.h"
#include "HRFKDirectRayleighRefined.h"
#include "HRFKDirectRayleighVertical.h"
#include "HRFKDirectRadial.h"
#include "HRFKDirectLove.h"
#include "SingleDirectRayleigh.h"
#include "ActiveConvFKVertical.h"
#include "ActiveHRFKVertical.h"
#include "ActiveHRFKRayleigh.h"
#include "ActiveHRFKDirectRayleigh.h"
#include "ActiveConvFKRayleigh.h"

namespace ArrayCore {

  /*!
    \class AbstractFKFunction AbstractFKFunction.h
    \brief AbstractFKFunction map calculator

    Compute the AbstractFKFunction map for a given array of stations.
  */

  /*!
    Init unlimited boundaries for wavenumber spaces (array, slowness and total)
    Null Gaussian weights.
  */
  AbstractFKFunction::AbstractFKFunction(FKCache * gridCache)
  {
    _steering=new FKSteeringOneComponent;
    _steering->setArray(gridCache->array());
    static_cast<FKSteeringOneComponent *>(_steering)->resize();

    _crossSpectrum=nullptr;
    _gridCache=gridCache;

    _squaredKmaxArray=0.0;
    _squaredKminArray=0.0;
    _squaredSlowMin=0.0;
    _squaredSlowMax=std::numeric_limits<double>::infinity();
    _squaredKminTotal=0.0;
    _squaredKmaxTotal=std::numeric_limits<double>::infinity();
    _minAzimuth=0.0;
    _maxAzimuth=2.0*M_PI;
  }

  AbstractFKFunction::~AbstractFKFunction()
  {
    delete _steering;
  }

  /*void AbstractFKFunction::setGrid(FKGradientSearch * g, double cacheStep, double step, double size)
  {
    g->setScales(Point(step, step, 1.0));
    g->setGrid(Point(size, size, 0.0));
  }*/

  void AbstractFKFunction::setGrid(FKGridSearch * g, double step, double size)
  {
    g->setGrid(-size, size, step, -size, size, step);
    setKxKySteering(g);
  }

  void AbstractFKFunction::setKxKySteering(FKGridSearch * g)
  {
    if(_gridCache->isEmpty(FKCache::OneComponent)) {
      App::log(tr("Caching steering vectors for 1-component ... (%1 values)\n").arg(g->nxy()));
      _gridCache->resize(FKCache::OneComponent, g->nxy());
      _gridCache->reportCacheVolume(FKCache::OneComponent, g->nxy(),
                                    FKSteeringOneComponent::sizeFactor);
      for(int i=g->nxy()-1; i>=0; i--) {
        FKSteeringOneComponent::init(_gridCache, i, g->coordinates(i));
      }
    }
  }

  /*!
    Angles are in degrees counted from North.
    \a min can be less that \a max. The range starts at \a min and is counted
    clockwise to \a max, eventually passing over 360 deg.

    Examples:

    \li min=10, max=270: all angles from 10 to 270 are accepted.
    \li min=270, max=10: all angles from 270 to 360 and 0 to 10 are accepted

  */
  void AbstractFKFunction::setAzimuthRange(double min, double max)
  {
    // Forces angles between 0 and 360 deg.
    min=Angle::canonicalDegrees(min);
    max=Angle::canonicalDegrees(max);
    // Angles are swapped because math range is counted counter-clockwise
    _maxAzimuth=Angle::geographicToMath(min);
    _minAzimuth=Angle::geographicToMath(max);
    if(_maxAzimuth<=_minAzimuth) {
      _maxAzimuth+=2.0*M_PI;
    }
    App::log(4, tr("Math angle range for FK map: from %1 to %2\n")
             .arg(_minAzimuth).arg(_maxAzimuth));
  }

  void AbstractFKFunction::setTotalLimits(double squaredOmega)
  {
    _squaredKmaxTotal=squaredOmega*_squaredSlowMax;
    if(_squaredKmaxTotal>_squaredKmaxArray) {
      _squaredKmaxTotal=_squaredKmaxArray;
    }
    _squaredKminTotal=squaredOmega*_squaredSlowMin;
    if(_squaredKminTotal<_squaredKminArray) {
      _squaredKminTotal=_squaredKminArray;
    }
  }

  /*!
    Return true if \a k is close to the limits (<1% relative).
    \a frequency is used only for reporting.
  */
  bool AbstractFKFunction::isOnLimit(const Point& k, double frequency) const
  {
    double k2=k.x()*k.x()+k.y()*k.y();
    // 1% on squared k corresponds to 0.5% on slowness if the k limit is controled by a maximum slowness
    if(fabs(k2-minimumK2())<0.01*minimumK2() ||
       fabs(k2-maximumK2())<0.01*maximumK2()) {
      APP_LOG(4, tr("  %1 (rad/m)^2 (%5 m/s) at %2 Hz on limits %3 or %4\n")
                          .arg(k2).arg(frequency)
                          .arg(minimumK2()).arg(maximumK2())
                          .arg(2.0*M_PI*frequency/sqrt(k2)))
      return true;
    } else {
      if(_minAzimuth+2.0*M_PI!=_maxAzimuth) { // If all angles are accepted, there is no limit
        double a=k.azimuth();
        return fabs(Angle::differenceRadians(a, _minAzimuth))<M_PI/180.0 ||
               fabs(Angle::differenceRadians(a, _maxAzimuth))<M_PI/180.0;
      } else {
        return false;
      }
    }
  }

  bool AbstractFKFunction::refine(const Point2D& k, double ell,
                                  FKPeaks * results,
                                  const WaveNumberConverter& conv)
  {
    Q_UNUSED(k)
    Q_UNUSED(ell)
    Q_UNUSED(results)
    Q_UNUSED(conv)
    return false;
  }

  /*Point AbstractFKFunction::gradient(const Point& k) const
  {
    if(isInsideLimits(k)) {
      FKPower p(_steering);
      setGradient(k, p);
      return p.gradient();
    } else {
      return 0.0;
    }
  }

  double AbstractFKFunction::concavity(const Point& k) const
  {
    if(isInsideLimits(k)) {
      FKPower p(_steering);
      p.initOneComponent();  // does not hurt for three components
      setGradient(k, p);
      setHessian(k, p);
      return p.concavity();
    } else {
      return 0.0;
    }
  }

  Point AbstractFKFunction::stepDirection(const Point& k) const
  {
    if(isInsideLimits(k)) {
      FKPower p(_steering);
      p.initOneComponent();  // does not hurt for three components
      setGradient(k, p);
      setHessian(k, p);
      if(p.concavity()<0.0) {
        return p.newtonStep();
      }
    }
    return Point();
  }*/

  AbstractFKFunction * AbstractFKFunction::create(ArrayStations::Mode mode,
                                                  const FKParameters * param,
                                                  FKCache * gridCache)
  {
    TRACE;
    AbstractFKFunction * f=nullptr;
    // Check that the components are available for the processing
    switch(mode) {
    case ArrayStations::Vertical:
      switch(param->processType()) {
      case FKParameters::Conventional:
      case FKParameters::Capon:
      case FKParameters::Omni:
      case FKParameters::ActiveCapon:
      case FKParameters::ActiveConventional:
        break;
      default:
        App::log(tr("'%1' processing is not allowed for vertical components.\n")
                 .arg(param->processTypeString()));
        break;
      }
      break;
    case ArrayStations::Horizontal:
      switch(param->processType()) {
      case FKParameters::ConventionalRadial:
      case FKParameters::ConventionalTransverse:
      case FKParameters::CaponRadial:
      case FKParameters::CaponTransverse:
      case FKParameters::LDS:
        break;
      default:
        App::log(tr("'%1' processing is not allowed for horizontal components.\n")
                 .arg(param->processTypeString()));
        break;
      }
      break;
    case ArrayStations::ThreeComponents:
      // Everything can be done with three components
      break;
    }
    switch(param->processType()) {
    // Vertical components
    case FKParameters::Conventional:
      f=new ConvFKVertical(gridCache);
      break;
    case FKParameters::Capon:
      f=new HRFKVertical(gridCache);
      break;
    case FKParameters::Omni:
      f=new OmniFKVertical(gridCache);
      break;
    case FKParameters::ActiveCapon:
      f=new ActiveHRFKVertical(gridCache);
      break;
    case FKParameters::ActiveConventional:
      f=new ActiveConvFKVertical(gridCache);
      break;
    // Horizontal components
    case FKParameters::ConventionalRadial:
      f=new ConvFKRadial(gridCache);
      break;
    case FKParameters::ConventionalTransverse:
      f=new ConvFKTransverse(gridCache);
      break;
    case FKParameters::CaponRadial:
      f=new HRFKRadial(gridCache);
      break;
    case FKParameters::CaponTransverse:
      f=new HRFKTransverse(gridCache);
      break;
    case FKParameters::LDS:
      f=new HRFKDirectLove(gridCache);
      break;
    // Three components
    case FKParameters::ConventionalRayleigh:
      f=new ConvFKRayleigh(gridCache);
      break;
    case FKParameters::RTBF:
      f=new HRFKRayleigh(gridCache);
      break;
    case FKParameters::RTBFRadial:
      f=new HRFKRayleighRadial(gridCache);
      break;
    case FKParameters::RTBFFixedEll:
      if(param->fixedEllipticityFileName().isEmpty()) {
        App::log(tr("Ellipticity file is not available in parameters.\n"));
      } else {
        HRFKRayleighFixedEll * f=new HRFKRayleighFixedEll(gridCache);
        if(!f->loadFixedEllipticityCurve(param->fixedEllipticityFileName())) {
          App::log(tr("Error loading ellipticity file '%1'.\n")
                   .arg(param->fixedEllipticityFileName()));
          delete f;
          f=nullptr;
        }
      }
      break;
    case FKParameters::RDS:
      f=new HRFKDirectRayleigh(gridCache);
      break;
    case FKParameters::RDSRadial:
      f=new HRFKDirectRayleighRadial(gridCache);
      break;
    case FKParameters::RDSVertical:
      f=new HRFKDirectRayleighVertical(gridCache);
      break;
    case FKParameters::RDSRefined:
      f=new HRFKDirectRayleighRefined(gridCache);
      break;
    case FKParameters::RDSSingle:
      f=new SingleDirectRayleigh(gridCache);
      break;
    case FKParameters::RDSFixedEll:
      if(param->fixedEllipticityFileName().isEmpty()) {
        App::log(tr("Ellipticity file is not available in parameters.\n"));
      } else {
        HRFKDirectRayleighFixedEll * f=new HRFKDirectRayleighFixedEll(gridCache);
        if(!f->loadFixedEllipticityCurve(param->fixedEllipticityFileName())) {
          App::log(tr("Error loading ellipticity file '%1'.\n")
                   .arg(param->fixedEllipticityFileName()));
          delete f;
          f=nullptr;
        }
      }
      break;
    //case FKParameters::Omni:  // TODO: Omni should be split vertical/three component
    //  f=new OmniFKRayleigh(gridCache);
    //  break;
    case FKParameters::PoggiVertical:
      f=new PoggiFKVertical(gridCache);
      break;
    case FKParameters::PoggiRadial:
      f=new PoggiFKRadial(gridCache);
      break;
    case FKParameters::ActiveConventionalRayleigh:
      f=new ActiveConvFKRayleigh(gridCache);
      break;
    case FKParameters::ActiveRTBF:
      f=new ActiveHRFKRayleigh(gridCache);
      break;
    case FKParameters::ActiveDS:
      f=new ActiveHRFKDirectRayleigh(gridCache);
      break;
    }
    if(f) {
      // Limiting search to kmin speedup computation but may pick alias peaks
      // Better to filter out after FK search
      //f->setMinimumWavenumber(param->minimumWaveNumber());
      // Reduce the gridded box to a circle of radius effective grid size
      f->setMaximumWavenumber(param->effectiveGridSize());
      f->setMaximumSlowness(param->maximumSlowness());
    }
    return f;
  }

} // namespace ArrayCore
