/***************************************************************************
**
**  This file is part of waran.
**
**  waran is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  waran is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2011-11-23
**  Copyright: 2011-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "CubeLink.h"
#include "Station.h"

/*!
  \class CubeLink CubeLink.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

CubeLink::CubeLink(Station * station)
  : DaemonLink(station)
{
  TRACE;
  setPort(2976);
  _refresh.setInterval(2000);
  QObject::connect(&_refresh, SIGNAL(timeout()), this, SLOT(refreshVariableParameters()));
  _refresh.start();
}

void CubeLink::connected()
{
  TRACE;
  DaemonLink::connected();
  requestParameters();
}

void CubeLink::refreshVariableParameters()
{
  if(isAvailable()) {
    requestParameters();
  }
}

void CubeLink::highRateRefresh()
{
  _refresh.setInterval(2000); // 2 s during startup
}

/*!

*/
int CubeLink::bytesAvailable(const char * buffer, int bytesCount)
{
  TRACE;
  int bytesRead=0;
  while(bytesRead<bytesCount) {
    // Scan for a line
    int newBytesRead;
    for(newBytesRead=bytesRead; newBytesRead<bytesCount && buffer[newBytesRead]!='\n'; newBytesRead++) {}
    if(newBytesRead==bytesCount) {
      return bytesRead; // partial line
    }
    switch(buffer[bytesRead]) {
    case 'f':
      if(match(buffer, bytesRead, bytesCount, "frequency=")) {
        frequency(buffer, bytesRead, bytesCount);
      }
      break;
    case 'h':
      if(match(buffer, bytesRead, bytesCount, "hardware gain level=")) {
        hardwareGain(buffer, bytesRead, bytesCount);
      }
      break;
    case 'r':
      if(match(buffer, bytesRead, bytesCount, "resolution=")) {
        numberOfBits(buffer, bytesRead, bytesCount);
      }
      break;
    case 's':
      if(match(buffer, bytesRead, bytesCount, "state=")) {
        state(buffer, bytesRead, newBytesRead);
      }
      break;
    default:
      break;
    }
    bytesRead=newBytesRead+1;
    // Skip blanks and additionnal end of line characters
    while(bytesRead<bytesCount && isspace(buffer[bytesRead])) {bytesRead++;}
  }
  return bytesRead;
}

void CubeLink::frequency(const char * buffer, int bytesRead, int bytesCount)
{
  TRACE;
  bool ok;
  int value=readInteger(buffer, bytesRead, bytesCount, ok);
  if(ok) {
    station()->setFrequency(value);
  }
}

void CubeLink::hardwareGain(const char * buffer, int bytesRead, int bytesCount)
{
  TRACE;
  bool ok;
  int value=readInteger(buffer, bytesRead, bytesCount, ok);
  if(ok) {
    station()->setHardwareGain(value);
  }
}

void CubeLink::numberOfBits(const char * buffer, int bytesRead, int bytesCount)
{
  TRACE;
  bool ok;
  double value=readDouble(buffer, bytesRead, bytesCount, ok);
  if(ok) {
    station()->setNumberOfBits(value);
  }
}

void CubeLink::state(const char * buffer, int bytesRead, int bytesCount)
{
  TRACE;
  if(match(buffer, bytesRead, bytesCount, "recording ")) {
    bool ok;
    long int value=readLongInteger(buffer, bytesRead, bytesCount, ok);
    if(value>0 && _refresh.interval()<10000) { // Reduce refresh rate (once every 10 s)
      _refresh.setInterval(10000);
    }
    if(ok) {
      station()->setSeismicState(QString("recording %1").arg(Number::secondsToDuration(value, 0)));
    }
  } else {
    station()->setSeismicState(QByteArray(buffer+bytesRead, bytesCount-bytesRead));
  }
}

void CubeLink::start()
{
  TRACE;
  send("start\n");
}

void CubeLink::stop()
{
  TRACE;
  send("stop\n");
}

void CubeLink::requestParameters()
{
  TRACE;
  send("frequency?\n"
       "gains?\n"
       "state?\n");
}

void CubeLink::setFrequency(double f)
{
  TRACE;
  QByteArray buf;
  buf+="frequency=";
  buf+=QByteArray::number((int)round(f));
  buf+="\n";
  send(buf);
}

void CubeLink::setGains(int hard, int soft)
{
  TRACE;
  QByteArray buf;
  buf+="gains=";
  buf+=QByteArray::number(hard);
  buf+=" ";
  buf+=QByteArray::number(soft);
  buf+="\n";
  send(buf);
}
