/***************************************************************************
**
**  This file is part of gpviewmax.
**
**  gpviewmax is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpviewmax is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-07-16
**  Copyright: 2018-2019

**
***************************************************************************/

#include <SciFigs.h>

#include "gpviewmaxVersion.h"
#include "gpviewmaxInstallPath.h"
#include "HistogramsWidget.h"
#include "Reader.h"

PACKAGE_INFO("gpviewmax", GPVIEWMAX)

ApplicationHelp * help();

HistogramsWidget * createWidgets(Reader& reader)
{
  HistogramsWidget * w=new HistogramsWidget;
  w->setNormalize(reader.isNormalized());
  w->setSamples(*reader.samples(), reader);
  w->setHistogramComments(reader.fileNames().join(";"));
  if(reader.fileNames().count()==1) {
    QFileInfo fi(reader.fileNames().first());
    w->setWindowTitle(fi.fileName()+" - gpviewmax");
  }
  if(!w->splitModes(*reader.samples(), reader)) {
    delete w;
    return nullptr;
  }
  return w;
}

int main(int argc, char ** argv)
{
  Application a(argc, argv, help);

  // Options
  ExportOptions exportOptions;
  Reader reader;
  if(!reader.setOptions(argc, argv)) {
    return 2;
  }
  if(!exportOptions.read(argc, argv)) {
    return 2;
  }
  if(!CoreApplication::checkRemainingArgs(argc, argv)) {
    return 2;
  }
  reader.setSamples(new Samples);
  if(!reader.read(argc, argv, reader.ignoreStdin())) {
    return 2;
  }
  if(reader.isEmpty()) {
    if(!reader.read(Message::getOpenFileNames(tr("Open data file"),
                                              tr("All files (*);;"
                                                 "Max files (*.max)")))) {
      return 2;
    }
  }
  // User basic filtering
  Samples * samples;
  if(!reader.undefinedEllipticities()) {
    samples=reader.samples()->filterUndefinedEllipticities();
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.curveFileNames().isEmpty()) {
    samples=reader.samples()->filterNoiseSmart(reader.noiseDeviation(),
                                               reader.slownessDeviation());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  samples=reader.samples()->filterNoise(reader.noiseThreshold());
  if(samples) {
    delete reader.samples();
    reader.setSamples(samples);
  }
  samples=reader.samples()->filterRelativePower(reader.relativePowerThreshold());
  if(samples) {
    delete reader.samples();
    reader.setSamples(samples);
  }
  samples=reader.samples()->filterMinimumVelocity(reader.minimumVelocity());
  if(samples) {
    delete reader.samples();
    reader.setSamples(samples);
  }
  samples=reader.samples()->filterMaximumVelocity(reader.maximumVelocity());
  if(samples) {
    delete reader.samples();
    reader.setSamples(samples);
  }
  samples=reader.samples()->filterMinimumWavenumber(reader.minimumWavenumber());
  if(samples) {
    delete reader.samples();
    reader.setSamples(samples);
  }
  samples=reader.samples()->filterMaximumWavenumber(reader.maximumWavenumber());
  if(samples) {
    delete reader.samples();
    reader.setSamples(samples);
  }

  if(!reader.setDefaultLimits()) {
    return 2;
  }

  int appReturn;
  switch(reader.action()) {
  case HistogramReader::Gui: {
      SciFigsGlobal s;
      HistogramsWidget * w=createWidgets(reader);
      if(!w) {
        return 2;
      }
      w->exportPlot(exportOptions);
      w->show();
      appReturn=a.exec();
      delete w;
    }
    break;
  case HistogramReader::MaxFile: {
      reader.samples()->save(reader.exportMaxFileName(), reader.headerLine());
      appReturn=0;
    }
    break;
  case HistogramReader::MinimumVelocity: {
      Histogram2D * h=reader.histogram(reader.dispersionSampling());
      h->setYScaleType(Scale::InversedLog);
      h->setSamples(reader.samples()->dispersionSamples());

      Curve<RealStatisticalPoint> c=h->pickAll(4, 0.1, 0.2);
      QList<Curve<RealStatisticalPoint>> l=c.split(1.2, LogScale, 0.1, 5);
      int c0=0, i0=0;
      double s0=0.0;
      for(int j=l.count()-1; j>=0; j--) {
        const Curve<RealStatisticalPoint>& c=l.at(j);
        int n=c.count();
        for(int i=0; i<n; i++) {
          const RealStatisticalPoint& p=c.constAt(i);
          double s=::exp(::log(p.mean())+2.0*::log(p.stddev()));
          if(s>s0) {
            c0=j;
            i0=i;
            s0=s;
          }
        }
      }
      QTextStream(stdout) << tr("Minimum velocity at %1 Hz (curve %2 point %3) = %4 m/s (mean-2*stddev on a log scale)\n")
                             .arg(l.at(c0).x(i0)).arg(c0).arg(i0).arg(1.0/s0);
      appReturn=0;
    }
    break;
  case HistogramReader::Plot: {
      SciFigsGlobal s;
      HistogramsWidget * w=createWidgets(reader);
      if(!w) {
        return 2;
      }
      w->exportPlot(exportOptions);
      appReturn=0;
      delete w;
    }
    break;
  default:
    appReturn=2;
    break;
  }
  return appReturn;
}

ApplicationHelp * help()
{
  TRACE;
  ApplicationHelp * h=new ApplicationHelp;
  h->setOptionSummary( "[OPTIONS] <FILE>...");
  h->setComments("Plot all values of a .max file generated with three-component FK. "
                 "Plot of individual modes.");
  h->addGroup("Main", "main");
  h->addOption("-R", "Plots Rayleigh ellipticity and noise, null ellipticities are filtered "
                     "out unless option '-undef-ell' is specified (default=automatically detected).");
  h->addOption("-no-split-ell-sign", "Do not split postive and negative ellipticities.");
  h->addOption("-undef-ell", "Keep samples with undefined ellipticities (default=filter them).");
  h->addGroup("filter", "filter");
  h->addOption("-rel-power <R>", "Accept only samples with power > R*maximum power (over a time window and for a frequency).");
  h->addOption("-max-noise <R>", "Accept only samples with relative incoherent noise < R.");
  h->addOption("-curve-rel-range <R>", "Keep R*mean around curves to split modes (default=0.2).");
  h->addOption("-low-noise <N> <S>", "Keep only low noise samples in a slowness neighborhood. "
                                     "For a given frequency, each sample S0, N0 is considered individually. "
                                     "If there is at least one sample Si, Ni in the slowness neighborhood "
                                     "defined by [(1-S)*S0, (1+S)*S0] for which N0 is larger than (1+N)*Ni, "
                                     "the current sample is filtered out.");
  h->addOption("-v-min <V>", "Filter out all samples have a velocity below V.");
  h->addOption("-v-max <V>", "Filter out all samples have a velocity above V.");
  h->addOption("-k-min <K>", "Filter out all samples have a wavenumber below K.");
  h->addOption("-k-max <K>", "Filter out all samples have a wavenumber above K.");
  h->addGroup("Output", "output");
  h->addOption("-plot", "Export plot (see '-h plot' for options).");
  h->addOption("-max <FILE>", "Output filter max values and exit.");
  h->addOption("-min-velocity", "Output minimum velocity decreased by two standard deviations and exit.");
  HistogramReader::helpCurves(h);
  HistogramReader::helpX(h);
  Reader::helpY(h);
  h->addGroup("Plot", "plot", 2);
  ExportOptions::help(h);
  return h;
}
