/***************************************************************************
**
**  This file is part of gptarget.
**
**  gptarget is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gptarget is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-28
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <DinverCore.h>
#include <DinverDCCore.h>
#include <QGpCoreTools.h>
#include <QGpCoreWave.h>
#include "TargetExtract.h"

/*!
  \class TargetExtract TargetExtract.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

TargetExtract::TargetExtract()
{
  _curveIndex=0;
  _ringIndex=0;
  _mode=Undefined;
}

/*!
  Description still missing
*/
bool TargetExtract::setOptions(int& argc, char ** argv)
{
  int j=1;
  for(int i=1; i<argc; i++) {
    QByteArray arg=argv[i];
    if(arg[0]=='-') {
      if(arg=="-disp") {
        _mode=Dispersion;
        CoreApplication::checkOptionArg(i, argc, argv);
        _curveIndex=CoreApplication::toInt(i, i-1, argv);
        if(_curveIndex<0) {
          App::log(tr("gptarget: curve index must be positive, see -help\n") );
          return false;
        }
      } else if(arg=="-autocorr") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _curveIndex=CoreApplication::toInt(i, i-1, argv);
          if(_curveIndex<0) {
            App::log(tr("gptarget: curve index must be positive, see -help\n") );
            return false;
          }
      } else if(arg=="-ring") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _ringIndex=CoreApplication::toInt(i, i-1, argv);
          if(_ringIndex<0) {
            App::log(tr("gptarget: ring index must be positive, see -help\n") );
            return false;
          }
      } else {
        App::log(tr("gptarget: bad option %1, see -help\n").arg(argv[i]) );
        return false;
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j < argc) {
    argv[j]=nullptr;
    argc=j;
  }
  if(_mode==Undefined) {
    App::log(tr("gptarget: missing ouput specification, see '-h all'\n") );
    return false;
  }
  if(argc!=2) {
    App::log(tr("gptarget: missing target file, see '-h all'\n") );
    return false;
  }
  setFileName(argv[1]);
  return true;
}

/*!
  Description still missing
*/
int TargetExtract::execute()
{
  TargetList tl;
  XMLVirtualPlugin plugin(&tl, "DispersionCurve");
  XMLDinverHeader hdr(&plugin);
  if(hdr.xml_restoreFile(fileName())!=XMLClass::NoError) {
    App::log(tr("gptarget: error reading file %1\n").arg(fileName()) );
    return 2;
  }
  QTextStream sOut(stdout);
  sOut << "# position " << tl.position().toString(6) << endl;
  switch(_mode) {
  case Dispersion:
    if(_curveIndex<tl.dispersionTarget().curves().count()) {
      const ModalCurve& c=tl.dispersionTarget().curves().at(_curveIndex);
      sOut << "# index " << _curveIndex << ": "
           << c.modes().first().toString(QGpCoreWave::Mode::Dispersion) << "\n"
           << c.toString();
    } else {
      App::log(tr("gptarget: curve index out of bounds (%1>%2)\n").arg(_curveIndex).arg(tl.dispersionTarget().curves().count()-1) );
      return 2;
    }
    break;
  case Autocorr: {
      const AutocorrCurves& cr=tl.autocorrTarget().curves();
      if(_ringIndex<cr.rings()->count()) {
        QList<ModalCurve> mc=cr.ringCurves(_ringIndex);
        if(_curveIndex<mc.count()) {
          const ModalCurve& c=mc.at(_curveIndex);
          sOut << "# index " << _curveIndex << ": "
               << c.modes().first().toString(QGpCoreWave::Mode::Autocorr) << "\n"
               << "# ring index " << _ringIndex << ": "
               << cr.ring(_ringIndex).toUserString() << "\n"
               << c.toString();
        } else {
          App::log(tr("gptarget: curve index out of bounds (%1>%2)\n").arg(_curveIndex).arg(mc.count()-1) );
          return 2;
        }
      } else {
        App::log(tr("gptarget: ring index out of bounds (%1>%2)\n").arg(_ringIndex).arg(cr.rings()->count()-1) );
        return 2;
      }
    }
    break;
  case Ellipticity:
  case RefraVp:
  case RefraVs:
  case Undefined:
    break;
  }
  return 0;
}
