/***************************************************************************
**
**  This file is part of geopsy-spac.
**
**  geopsy-spac is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  geopsy-spac is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2019-02-01
**  Copyright: 2019

**
***************************************************************************/

#include <signal.h>
#include <ArrayCore.h>

#include "geopsy-spacVersion.h"
#include "geopsy-spacInstallPath.h"

PACKAGE_INFO("geopsy-spac", GEOPSY_SPAC)

ApplicationHelp * help();
void showProgress(int);

SPACTool * tool=nullptr;

bool postOptions(int argc, char ** argv, SPACParameters * param)
{
  int i, j = 1;
  for (i=1; i<argc; i++) {
    QByteArray arg = argv[i];
    if (arg[0]=='-') {
      if(arg=="-param-example") {
        QTextStream(stdout) << param->AbstractParameters::toString()
                            << param->toString();
        ::exit(0);
      } else {
        argv[j++]=argv[i];
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  return true;
}

int main(int argc, char ** argv)
{
  CoreApplication a(argc, argv, help);
  // No plugins are loaded, TODO: pre-parse options to activate plugins
  GeopsyCoreEngine gp(false);
  skipOpenBlasMultiThreading();
#ifdef Q_OS_WIN
  signal(30, showProgress);  // From the list available with 'kill -l'
#else
  signal(SIGUSR1, showProgress);
#endif

  SPACParameters * param=new SPACParameters;
  CoreToolInterface gti;
  gti.setParameters(param);
  if(!gti.setOptions(argc, argv)) {
    return 2;
  }
  if(!postOptions(argc, argv, param)) {
    return 2;
  }
  if(!CoreApplication::checkRemainingArgs(argc, argv)) {
    return 2;
  }
  if(!gti.outputBaseName().isEmpty()) {
    param->setOutputBaseName(gti.outputBaseName());
  }
  int n=gti.subPoolCount();
  for(int i=0; i<n; i++) {
    tool=new SPACTool;
    SubSignalPool array=gti.subPool(i);
    if(!tool->setSubPool(&array)) {
      App::log(tr("Error initializing group '%1'\n").arg(array.name()));
      return 2;
    }
    if(!tool->setParameters(*param)) {
      tool->waitFinished();
      // K max solver not properly deleted without an event loop...
      return 2;
    }
    tool->start();
    tool->waitFinished();
    tool->results()->commitStatistics();
    tool->results()->reportRejects();
    tool->results()->save(param->outputName(gti.groupName(i)), tool->log());
    delete tool;
    tool=nullptr;
    gp.cache()->free(); // Avoid to keep all signals from all groups in memory
  }
  return 0;
}

void showProgress(int)
{
  if(tool) {
    tool->showProgress();
  }
}

ApplicationHelp * help()
{
  TRACE;
  ApplicationHelp * h=new ApplicationHelp;
  h->setOptionSummary( "[OPTIONS] [SIGNAL_FILES]");
  h->setComments("Calculate SPAC for ambient vibrations.\n"
                 "Sending signal SIGUSR1 prints the current progress.");
  h->addGroup("geopsy-spac", "geopsy-spac");
  h->addOption("-param-example","Output default parameters, template to build a new parameter file.");
  CoreToolInterface::setHelp(h);
  return h;
}
