/***************************************************************************
**
**  This file is part of dinverdc.
**
**  dinverdc is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  dinverdc is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2005-10-31
**  Copyright: 2005-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <DinverCore.h>
#include <DinverDCCore.h>
#include <QGpGuiTools.h>

#include "TargetListWidget.h"
#include "ParamGroundModelWidget.h"
#include "ParamProfileWidget.h"
#include "DispersionTargetWidget.h"
#include "AutocorrTargetWidget.h"
#include "EllipticityCurveTargetWidget.h"
#include "EllipticityPeakTargetWidget.h"
#include "RefractionTargetWidget.h"
#include "MagnetoTelluricTargetWidget.h"

/*
 *  Constructs a TargetListWidget as a child of 'parent', with the
 *  name 'name' and widget flags set to 'f'.
 */
TargetListWidget::TargetListWidget(QWidget *parent, Qt::WindowFlags f)
    : QWidget(parent, f)
{
  TRACE;
  setupUi(this);

  _dispWidget=nullptr;
  _autocorrWidget=nullptr;
  _ellCurveWidget=nullptr;
  _ellPeakWidget=nullptr;
  _refraVpWidget=nullptr;
  _refraVsWidget=nullptr;
  _magnetoTelluricWidget=nullptr;

  updateInfo();

  // Temporary out of order
  refraVpSelect->setEnabled(false);
  refraVsSelect->setEnabled(false);
  magnetoTelluricSelect->setEnabled(false);
}

/*!
  \a name can be null to reset
*/
void TargetListWidget::setTitles(const QString& name)
{
  TRACE;
  if(!name.isEmpty()) {
    if(_dispWidget) _dispWidget->setWindowTitle(tr("Dispersion curve target of %1").arg(name));
    if(_autocorrWidget) _autocorrWidget->setWindowTitle(tr("Autocorrelation curve target of %1").arg(name));
    if(_ellCurveWidget) _ellCurveWidget->setWindowTitle(tr("Ellipticity curve target of %1").arg(name));
    if(_ellPeakWidget) _ellPeakWidget->setWindowTitle(tr("Ellipticity peak target of %1").arg(name));
    if(_refraVpWidget) _refraVpWidget->setWindowTitle(tr("Refraction Vp target of %1").arg(name));
    if(_refraVsWidget) _refraVsWidget->setWindowTitle(tr("Refraction Vs target of %1").arg(name));
    if(_magnetoTelluricWidget) _magnetoTelluricWidget->setWindowTitle(tr("Magneto-telluric target of %1").arg(name));
  } else {
    if(_dispWidget) _dispWidget->setWindowTitle(tr("Dispersion curve target"));
    if(_autocorrWidget) _autocorrWidget->setWindowTitle(tr("Autocorrelation curve target"));
    if(_ellCurveWidget) _ellCurveWidget->setWindowTitle(tr("Ellipticity curve target"));
    if(_ellPeakWidget) _ellPeakWidget->setWindowTitle(tr("Ellipticity peak target"));
    if(_refraVpWidget) _refraVpWidget->setWindowTitle(tr("Refraction Vp target"));
    if(_refraVsWidget) _refraVsWidget->setWindowTitle(tr("Refraction Vs target"));
    if(_magnetoTelluricWidget) _magnetoTelluricWidget->setWindowTitle(tr("Magneto-telluric target"));
  }
}

void TargetListWidget::on_dispSelect_toggled(bool)
{
  TRACE;
  bool b=dispSelect->isChecked();
  dispSet->setEnabled(b);
  dispWeight->setEnabled(b);
  dispMinMisfit->setEnabled(b);
  updateInfo();
}

void TargetListWidget::on_autocorrSelect_toggled(bool)
{
  TRACE;
  bool b=autocorrSelect->isChecked();
  autocorrSet->setEnabled(b);
  autocorrWeight->setEnabled(b);
  autocorrMinMisfit->setEnabled(b);
  updateInfo();
}

void TargetListWidget::on_ellCurveSelect_toggled(bool)
{
  TRACE;
  bool b=ellCurveSelect->isChecked();
  ellCurveSet->setEnabled(b);
  ellCurveWeight->setEnabled(b);
  ellCurveMinMisfit->setEnabled(b);
  updateInfo();
}

void TargetListWidget::on_ellPeakSelect_toggled(bool)
{
  TRACE;
  bool b=ellPeakSelect->isChecked();
  ellPeakSet->setEnabled(b);
  ellPeakWeight->setEnabled(b);
  ellPeakMinMisfit->setEnabled(b);
  updateInfo();
}

void TargetListWidget::on_refraVpSelect_toggled(bool)
{
  TRACE;
  bool b=refraVpSelect->isChecked();
  refraVpWeight->setEnabled(b);
  refraVpSet->setEnabled(b);
  refraVpMinMisfit->setEnabled(b);
  updateInfo();
}

void TargetListWidget::on_refraVsSelect_toggled(bool)
{
  TRACE;
  bool b=refraVsSelect->isChecked();
  refraVsWeight->setEnabled(b);
  refraVsSet->setEnabled(b);
  refraVsMinMisfit->setEnabled(b);
  updateInfo();
}

void TargetListWidget::on_magnetoTelluricSelect_toggled(bool)
{
  TRACE;
  bool b=magnetoTelluricSelect->isChecked();
  magnetoTelluricWeight->setEnabled(b);
  magnetoTelluricSet->setEnabled(b);
  magnetoTelluricMinMisfit->setEnabled(b);
  updateInfo();
}

void TargetListWidget::clear()
{
  TRACE;
  TargetList tl;
  setFrom(&tl);
}

void TargetListWidget::on_dispSet_clicked()
{
  TRACE;
  DispersionTargetWidget *& w=_dispWidget;
  if(w) {
    DinverCoreEngine::instance()->activateSubWindow(w);
  } else {
    w=new DispersionTargetWidget;
    w->setObjectName("dispersionTarget");
    DinverCoreEngine::instance()->addSubWindow(this, w)->setUserClosable(false);
    connect(w , SIGNAL(curveModified()), this, SLOT(updateInfo()));
  }
}

void TargetListWidget::on_autocorrSet_clicked()
{
  TRACE;
  AutocorrTargetWidget *& w=_autocorrWidget;
  if(w) {
    DinverCoreEngine::instance()->activateSubWindow(w);
  } else {
    w=new AutocorrTargetWidget;
    w->setObjectName("autocorrCurveTarget");
    DinverCoreEngine::instance()->addSubWindow(this, w)->setUserClosable(false);
    connect(w, SIGNAL(curveModified()), this, SLOT(updateInfo()));
  }
}

void TargetListWidget::on_ellCurveSet_clicked()
{
  TRACE;
  EllipticityCurveTargetWidget *& w=_ellCurveWidget;
  if(w) {
    DinverCoreEngine::instance()->activateSubWindow(w);
  } else {
    w=new EllipticityCurveTargetWidget;
    w->setObjectName("ellipticityCurveTarget");
    DinverCoreEngine::instance()->addSubWindow(this, w)->setUserClosable(false);
    connect(w, SIGNAL(curveModified()), this, SLOT(updateInfo()));
  }
}

void TargetListWidget::on_ellPeakSet_clicked()
{
  TRACE;
  EllipticityPeakTargetWidget *& w=_ellPeakWidget;
  if(w) {
    DinverCoreEngine::instance()->activateSubWindow(w);
  } else {
    w=new EllipticityPeakTargetWidget;
    w->setObjectName("ellipticityPeakTarget");
    DinverCoreEngine::instance()->addSubWindow(this, w)->setUserClosable(false);
    connect(w, SIGNAL(peakModified()), this, SLOT(updateInfo()));
  }
}

void TargetListWidget::on_refraVpSet_clicked()
{
  TRACE;
  RefractionTargetWidget *& w=_refraVpWidget;
  if(w) {
    DinverCoreEngine::instance()->activateSubWindow(w);
  } else {
    w=new RefractionTargetWidget;
    w->setObjectName("refractionVpTarget");
    w->setWindowTitle(tr("Refraction Vp target"));
    DinverCoreEngine::instance()->addSubWindow(this, w)->setUserClosable(false);
    connect(w, SIGNAL(curveModified()), this, SLOT(updateInfo()));
  }
}

void TargetListWidget::on_refraVsSet_clicked()
{
  TRACE;
  RefractionTargetWidget *& w=_refraVsWidget;
  if(w) {
    DinverCoreEngine::instance()->activateSubWindow(w);
  } else {
    w=new RefractionTargetWidget;
    w->setObjectName("refractionVsTarget");
    w->setWindowTitle(tr("Refraction Vs target"));
    DinverCoreEngine::instance()->addSubWindow(this, w)->setUserClosable(false);
    connect(w, SIGNAL(curveModified()), this, SLOT(updateInfo()));
  }
}

void TargetListWidget::on_magnetoTelluricSet_clicked()
{
  TRACE;
  MagnetoTelluricTargetWidget *& w=_magnetoTelluricWidget;
  if(w) {
    DinverCoreEngine::instance()->activateSubWindow(w);
  } else {
    w=new MagnetoTelluricTargetWidget;
    w->setObjectName("magnetoTelluricTarget");
    w->setWindowTitle(tr("Magneto-telluric target"));
    DinverCoreEngine::instance()->addSubWindow(this, w)->setUserClosable(false);
    connect(w, SIGNAL(curveModified()), this, SLOT(updateInfo()));
  }
}

void TargetListWidget::updateInfo()
{
  TRACE;
  QString str;
  if(dispSelect->isChecked() && _dispWidget) {
    str+=tr("%1 dispersion curves").arg(_dispWidget->count());
  }
  if(autocorrSelect->isChecked() && _autocorrWidget) {
    if(!str.isEmpty()) str+=", ";
    str+=tr("%1 autocorelation curves").arg(_autocorrWidget->count());
  }
  if(ellCurveSelect->isChecked() && _ellCurveWidget) {
    if(!str.isEmpty()) str+=", ";
    str+=tr("%1 ellipticity curves").arg(_ellCurveWidget->count());
  }
  if(ellPeakSelect->isChecked() && _ellPeakWidget) {
    if(!str.isEmpty()) str+=", ";
    str+=tr("ellipticity peak at %1 Hz").arg(_ellPeakWidget->value().mean());
  }
  if(refraVpSelect->isChecked() && _refraVpWidget) {
    if(!str.isEmpty()) str+=", ";
    str+=tr("%1 refraction Vp curves").arg(_refraVpWidget->count());
  }
  if(refraVsSelect->isChecked() && _refraVsWidget) {
    if(!str.isEmpty()) str+=", ";
    str+=tr("%1 refraction Vs curves").arg(_refraVsWidget->count());
  }
  if(magnetoTelluricSelect->isChecked() && _magnetoTelluricWidget) {
    if(!str.isEmpty()) str+=", ";
    str+=tr("%1 magneto-telluric curves").arg(_magnetoTelluricWidget->count());
  }
  if(str.isEmpty()) str=tr("No target curve defined");
  info->setText(str);
  emit targetChanged();
}

void TargetListWidget::setParamProfiles(ParamGroundModelWidget * m)
{
  TRACE;
  // Add eventually new profiles
  if(dispSelect->isChecked() || ellCurveSelect->isChecked() || ellPeakSelect->isChecked() || autocorrSelect->isChecked()) {
    if( !m->find("Vp"))
      m->newProfile()->setParam("Vp", tr("Compression-wave velocity"), tr("m/s"), 200, 5000,
                                ParamProfile::Param, SimpleCondition::LessThan);
    if( !m->find("Nu"))
      m->newProfile()->setParam("Nu", tr("Poisson's Ratio"), tr(""), 0.2, 0.5,
                                ParamProfile::Condition, SimpleCondition::GreaterThan);
    if( !m->find("Vs"))
      m->newProfile()->setParam("Vs", tr("Shear-wave velocity"), tr("m/s"), 50, 3500,
                                ParamProfile::Param, SimpleCondition::LessThan);
    if( !m->find( "Rho" ))
      m->newProfile()->setParam("Rho", tr("Density"), tr("kg/m3"), 2000, 2000,
                                ParamProfile::Param, SimpleCondition::LessThan);
    //if(autocorrSelect->isChecked() && !m->find( "Alpha" ))
    //  m->newProfile()->setParam("Alpha", tr("Rayleigh/Horizontal energy ratio"), tr(""), 0, 1);
  }
  if(refraVpSelect->isChecked()) {
    if( !m->find("Vp")) {
      m->newProfile()->setParam("Vp", tr("Compression-wave velocity"), tr("m/s"), 200, 5000,
                                ParamProfile::Param, SimpleCondition::LessThan);
    }
    if( !m->find("Pitch")) {
      m->newProfile()->setParam("Pitch", tr("Layer pitch (from -1 to 1)"), tr(""), 0.0, 0.0,
                                ParamProfile::Param, SimpleCondition::LessThan);
    }
  }
  if(refraVsSelect->isChecked()) {
    if( !m->find("Vs")) {
      m->newProfile()->setParam("Vs", tr("Shear-wave velocity"), tr("m/s"), 150, 3500,
                                ParamProfile::Param, SimpleCondition::LessThan);
    }
    if(refraVpSelect->isChecked()) {
      if( !m->find("Nu")) {
        m->newProfile()->setParam("Nu", tr("Poisson's Ratio"), tr(""), 0.2, 0.5,
                                  ParamProfile::Condition, SimpleCondition::GreaterThan);
      }
    } else {
      if( !m->find("Pitch")) {
        m->newProfile()->setParam("Pitch", tr("Layer pitch (from -1 to 1)"), tr(""), 0.0, 0.0,
                                  ParamProfile::Param, SimpleCondition::LessThan);
      }
    }
  }
  if(magnetoTelluricSelect->isChecked()) {
     if( !m->find("Res")) {
       m->newProfile()->setParam("Res", tr("Resistivity"), tr("ohm m"), 1.0, 10000.0,
                                 ParamProfile::Param, SimpleCondition::NoCondition);
     }
  }
  // Remove useless profiles
  QList<ParamProfileWidget *> objList=m->modelSheet->widget() ->findChildren<ParamProfileWidget *>();
  for(QList<ParamProfileWidget *>::iterator it=objList.begin(); it!=objList.end(); ++it) {
    ParamProfileWidget * cur=* it;
    QString shortName=cur->shortName();
    bool accept=false;
    if(dispSelect->isChecked() || ellCurveSelect->isChecked() || ellPeakSelect->isChecked() || autocorrSelect->isChecked()) {
      if(shortName=="Vp" || shortName=="Nu" || shortName=="Vs" || shortName=="Rho") accept=true;
      //if(autocorrSelect->isChecked() && shortName=="Alpha") accept=true;
    }
    if(refraVpSelect->isChecked()) {
      if(shortName=="Vp" || shortName=="Pitch") accept=true;
    }
    if(refraVsSelect->isChecked()) {
      if(refraVpSelect->isChecked()) {
        if(shortName=="Nu") accept=true;
      }
      if(shortName=="Vs" || shortName=="Pitch") accept=true;
    }
    if(magnetoTelluricSelect->isChecked()) {
      if(shortName=="Res") accept=true;
    }
    if( !accept) delete cur;
  }
}

/*!
  If \a all is false, only the visible curves will be included in the target list.
*/
TargetList * TargetListWidget::targetList()
{
  TRACE;
  TargetList * tl=new TargetList;
  tl->dispersionTarget().setSelected(dispSelect->isChecked());
  tl->autocorrTarget().setSelected(autocorrSelect->isChecked());
  tl->ellipticityCurveTarget().setSelected(ellCurveSelect->isChecked());
  tl->ellipticityPeakTarget().setSelected(ellPeakSelect->isChecked());
  tl->refractionVpTarget().setSelected(refraVpSelect->isChecked());
  tl->refractionVsTarget().setSelected(refraVsSelect->isChecked());
  tl->magnetoTelluricTarget().setSelected(magnetoTelluricSelect->isChecked());

  if(_dispWidget) {
    tl->dispersionTarget().setCurves(_dispWidget->curves());
    tl->dispersionTarget().setMisfitWeight(dispWeight->text().toDouble());
    tl->dispersionTarget().setMinimumMisfit(dispMinMisfit->text().toDouble());
  }
  if(_autocorrWidget) {
    tl->autocorrTarget().setCurves(_autocorrWidget->curves());
    tl->autocorrTarget().setMisfitWeight(autocorrWeight->text().toDouble());
    tl->autocorrTarget().setMinimumMisfit(autocorrMinMisfit->text().toDouble());
  }
  if(_ellCurveWidget) {
    tl->ellipticityCurveTarget().setCurves(_ellCurveWidget->curves());
    tl->ellipticityCurveTarget().setMisfitWeight(ellCurveWeight->text().toDouble());
    tl->ellipticityCurveTarget().setMinimumMisfit(ellCurveMinMisfit->text().toDouble());
  }
  if(_ellPeakWidget) {
    tl->ellipticityPeakTarget().setValue(_ellPeakWidget->value());
    tl->ellipticityPeakTarget().setMinimumAmplitude(_ellPeakWidget->minimumAmplitude());
    tl->ellipticityPeakTarget().setMisfitWeight(ellPeakWeight->text().toDouble());
    tl->ellipticityPeakTarget().setMinimumMisfit(ellPeakMinMisfit->text().toDouble());
  }
  if(_refraVpWidget) {
    tl->refractionVpTarget().setCurves(_refraVpWidget->curves());
    tl->refractionVpTarget().setMisfitWeight(refraVpWeight->text().toDouble());
    tl->refractionVpTarget().setMinimumMisfit(refraVpMinMisfit->text().toDouble());
  }
  if(_refraVsWidget) {
    tl->refractionVsTarget().setCurves(_refraVsWidget->curves());
    tl->refractionVsTarget().setMisfitWeight(refraVsWeight->text().toDouble());
    tl->refractionVsTarget().setMinimumMisfit(refraVsMinMisfit->text().toDouble());
  }
  if(_magnetoTelluricWidget) {
    tl->magnetoTelluricTarget().setCurves(_magnetoTelluricWidget->curves());
    tl->magnetoTelluricTarget().setMisfitWeight(magnetoTelluricWeight->text().toDouble());
    tl->magnetoTelluricTarget().setMinimumMisfit(magnetoTelluricMinMisfit->text().toDouble());
  }
  tl->validateTargets();
  return tl;
}

void TargetListWidget::setFrom(TargetList * tl)
{
  TRACE;
  dispSelect->setChecked(tl->dispersionTarget().selected());
  autocorrSelect->setChecked(tl->autocorrTarget().selected());
  ellCurveSelect->setChecked(tl->ellipticityCurveTarget().selected());
  ellPeakSelect->setChecked(tl->ellipticityPeakTarget().selected());
  refraVpSelect->setChecked(tl->refractionVpTarget().selected());
  refraVsSelect->setChecked(tl->refractionVsTarget().selected());
  magnetoTelluricSelect->setChecked(tl->magnetoTelluricTarget().selected());

  if(!tl->dispersionTarget().curves().isEmpty()) {
    on_dispSet_clicked();
    ASSERT(_dispWidget);
    _dispWidget->clear();
    _dispWidget->addCurves(tl->dispersionTarget().curves());
    dispWeight->setText(QString::number(tl->dispersionTarget().misfitWeight()));
    dispMinMisfit->setText(QString::number(tl->dispersionTarget().minimumMisfit()));
  } else {
    if(_dispWidget) _dispWidget->clear();
    dispWeight->setText("1");
    dispMinMisfit->setText("0");
    dispSelect->setChecked(false);
  }

  if(!tl->autocorrTarget().curves().isEmpty()) {
    on_autocorrSet_clicked();
    ASSERT(_autocorrWidget);
    _autocorrWidget->clear();
    _autocorrWidget->addCurves(tl->autocorrTarget().curves());
    autocorrWeight->setText(QString::number(tl->autocorrTarget().misfitWeight()));
    autocorrMinMisfit->setText(QString::number(tl->autocorrTarget().minimumMisfit()));
  } else {
    if(_autocorrWidget) _autocorrWidget->clear();
    autocorrSelect->setChecked(false);
    autocorrWeight->setText("1");
    autocorrMinMisfit->setText("0");
    autocorrSelect->setChecked(false);
  }

  if(!tl->ellipticityCurveTarget().curves().isEmpty()) {
    on_ellCurveSet_clicked();
    ASSERT(_ellCurveWidget);
    _ellCurveWidget->clear();
    _ellCurveWidget->addCurves(tl->ellipticityCurveTarget().curves());
    ellCurveWeight->setText(QString::number(tl->ellipticityCurveTarget().misfitWeight()));
    ellCurveMinMisfit->setText(QString::number(tl->ellipticityCurveTarget().minimumMisfit()));
  } else {
    if(_ellCurveWidget) _ellCurveWidget->clear();
    ellCurveSelect->setChecked(false);
    ellCurveWeight->setText("1");
    ellCurveMinMisfit->setText("0");
    ellCurveSelect->setChecked(false);
  }

  if(tl->ellipticityPeakTarget().value().isValid()) {
    on_ellPeakSet_clicked();
    ASSERT(_ellPeakWidget);
    _ellPeakWidget->setValue(tl->ellipticityPeakTarget().value());
    _ellPeakWidget->setMinimumAmplitude(tl->ellipticityPeakTarget().minimumAmplitude());
    ellPeakWeight->setText(QString::number(tl->ellipticityPeakTarget().misfitWeight()));
    ellPeakMinMisfit->setText(QString::number(tl->ellipticityPeakTarget().minimumMisfit()));
  } else {
    ellPeakWeight->setText("1");
    ellPeakMinMisfit->setText("0");
    ellPeakSelect->setChecked(false);
  }

  if( !tl->refractionVpTarget().curves().isEmpty()) {
    on_refraVpSet_clicked();
    ASSERT(_refraVpWidget);
    _refraVpWidget->clear();
    _refraVpWidget->addCurves(tl->refractionVpTarget().curves());
    refraVpWeight->setText(QString::number(tl->refractionVpTarget().misfitWeight()));
    refraVpMinMisfit->setText(QString::number(tl->refractionVpTarget().minimumMisfit()));
  } else {
    if(_refraVpWidget) _refraVpWidget->clear();
    refraVpWeight->setText("1");
    refraVpMinMisfit->setText("0");
    refraVpSelect->setChecked(false);
  }

  if( !tl->refractionVsTarget().curves().isEmpty()) {
    on_refraVsSet_clicked();
    ASSERT(_refraVsWidget);
    _refraVsWidget->clear();
    _refraVsWidget->addCurves(tl->refractionVsTarget().curves());
    refraVsWeight->setText(QString::number(tl->refractionVpTarget().misfitWeight()));
    refraVsMinMisfit->setText(QString::number(tl->refractionVpTarget().minimumMisfit()));
  } else {
    if(_refraVsWidget) _refraVsWidget->clear();
    refraVsWeight->setText("1");
    refraVsMinMisfit->setText("0");
    refraVsSelect->setChecked(false);
  }

  if( !tl->magnetoTelluricTarget().curves().isEmpty()) {
    on_magnetoTelluricSet_clicked();
    ASSERT(_magnetoTelluricWidget);
    _magnetoTelluricWidget->clear();
    _magnetoTelluricWidget->addCurves(tl->magnetoTelluricTarget().curves());
    magnetoTelluricWeight->setText(QString::number(tl->magnetoTelluricTarget().misfitWeight()));
    magnetoTelluricMinMisfit->setText(QString::number(tl->magnetoTelluricTarget().minimumMisfit()));
  } else {
    if(_magnetoTelluricWidget) _magnetoTelluricWidget->clear();
    magnetoTelluricWeight->setText("1");
    magnetoTelluricMinMisfit->setText("0");
    magnetoTelluricSelect->setChecked(false);
  }

  updateInfo();
}

void TargetListWidget::setEditable(bool e)
{
  TRACE;
  dispWeight->setEnabled(e);
  dispMinMisfit->setEnabled(e);
  dispSelect->setEnabled(e);
  dispSet->setText(e ? tr("Set") : tr("View"));
  if(_dispWidget) _dispWidget->setEditable(e);

  autocorrWeight->setEnabled(e);
  autocorrMinMisfit->setEnabled(e);
  autocorrSelect->setEnabled(e);
  autocorrSet->setText(e ? tr("Set") : tr("View"));
  if(_autocorrWidget) _autocorrWidget->setEditable(e);

  ellCurveWeight->setEnabled(e);
  ellCurveMinMisfit->setEnabled(e);
  ellCurveSelect->setEnabled(e);
  ellCurveSet->setText(e ? tr("Set") : tr("View"));
  if(_ellCurveWidget) _ellCurveWidget->setEditable(e);

  ellPeakWeight->setEnabled(e);
  ellPeakMinMisfit->setEnabled(e);
  ellPeakSelect->setEnabled(e);
  ellPeakSet->setText(e ? tr("Set") : tr("View"));
  if(_ellPeakWidget) _ellPeakWidget->setEditable(e);

  refraVpWeight->setEnabled(e);
  refraVpMinMisfit->setEnabled(e);
  refraVpSelect->setEnabled(e);
  refraVpSet->setText(e ? tr("Set") : tr("View"));
  if(_refraVpWidget) _refraVpWidget->setEditable(e);

  refraVsWeight->setEnabled(e);
  refraVsMinMisfit->setEnabled(e);
  refraVsSelect->setEnabled(e);
  refraVsSet->setText(e ? tr("Set") : tr("View"));
  if(_refraVsWidget) _refraVsWidget->setEditable(e);

  magnetoTelluricWeight->setEnabled(e);
  magnetoTelluricMinMisfit->setEnabled(e);
  magnetoTelluricSelect->setEnabled(e);
  magnetoTelluricSet->setText(e ? tr("Set") : tr("View"));
  if(_magnetoTelluricWidget) _magnetoTelluricWidget->setEditable(e);
}
