/***************************************************************************
**
**  This file is part of dinverdc.
**
**  dinverdc is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  dinverdc is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2005-10-31
**  Copyright: 2005-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <DinverDCCore.h>
#include <SciFigs.h>

#include "DispersionTargetWidget.h"

/*!
 Constructs a DispersionTargetWidget as a child of \a parent and widget flags set to \a fl.
 DispersionTargetWidget basically includes a CurveBrowser to let the user select the
 target dispersion curves and input its current knowledge of the modal identification.
*/
DispersionTargetWidget::DispersionTargetWidget(QWidget* parent)
    : ModalCurveBrowser(parent)
{
  TRACE;
  AxisWindow * w=new AxisWindow(this);
  dock(w);
  setProxy(new DispersionProxy);
  setWindowTitle(tr("Dispersion target"));
  setModeType(Mode::Dispersion);
  setPlotProxy(new DispersionPlotProxy);
  LineLayer * l=new LineLayer(w);
  l->setErrorBar(LineLayer::VerticalLogBar);
  initLayer(l);
  setCurrentLayer(l);
}

DispersionTargetWidget::~DispersionTargetWidget()
{
  TRACE;
  WidgetRect::saveRect(this);
}

void DispersionTargetWidget::resizeEvent(QResizeEvent * e)
{
  TRACE;
  WidgetRect::record(this);
  ModalCurveBrowser::resizeEvent(e);
}

void DispersionTargetWidget::load()
{
  TRACE;
  MessageContext();
  QStringList fileNames=Message::getOpenFileNames(tr("Load curve to fit"),
                                                  tr("All files (*);;"
                                                     "dinver environment (*.dinver);;"
                                                     "dinver target (*.target);;"
                                                     "na_viewer reports (*.report)"));
  if(!fileNames.isEmpty()) {
    int curvesCount=count();
    ColumnTextParser * parser=nullptr;
    int n=fileNames.count();
    for(int i=0; i<n; i++) {
      QString fileName=fileNames.at(i);
      QFileInfo fi(fileName);
      if(fi.suffix()=="target" || fi.suffix()=="dinver") {
        loadTarget(fileName);
      } else if(fi.suffix()=="report") {
        loadReport(fileName);
      } else {
        loadMultiColumns(fileName, i==n-1, parser);
      }
    }
    delete parser;
    if(curvesCount==0) setLimits();
  }
}

void DispersionTargetWidget::loadTarget(QString fileName)
{
  TRACE;
  TargetList tl;
  XMLVirtualPlugin plugin(&tl, "DispersionCurve");
  XMLDinverHeader hdr(&plugin);
  if(hdr.xml_restoreFile(fileName)!=XMLClass::NoError) {
    Message::warning(MSG_ID, tr("Loading Dinver target/environment ..."),
                     tr("Error while reading file %1").arg(fileName), Message::cancel());
    return ;
  }
  addCurves(tl.dispersionTarget().curves());
}

void DispersionTargetWidget::loadReport(QString fileName)
{
  TRACE;
  QFileInfo finfo(fileName);
  fileName=finfo.absoluteFilePath();
  CompatInversionReport report(false, fileName);
  if(!report.isValid()) {
    return;
  }
  if(report.reportType()!=CompatInversionReport::DispersionGoal) {
    Message::warning(MSG_ID, tr("Importing report"),
                     tr("The report you selected contains no dispersion curve\n"), Message::cancel());
    return;
  }
  CompatDispersionData * dispersion=report.loadDispersionGoal();
  // Get the values for each mode
  int nModes=dispersion->modesCount();
  for(int i=0; i<nModes; i++) {
    addCurve(dispersion->curve(i));
  }
  delete dispersion;
}
