/***************************************************************************
**
**  This file is part of SciFigs.
**
**  SciFigs is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  SciFigs is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-01-22
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "PageSize.h"
#include "PageSizeProperties.h"

namespace SciFigs {

  const QString PageSize::xmlPageSizeTag="PageSize";

  /*!
    \class PageSize PageSize.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  PageSize::PageSize()
  {
    TRACE;
    setPaperSize(QPageSize::A4);
    _orientation=QPageLayout::Portrait;
  }

  /*!
    Description of constructor still missing
  */
  PageSize::PageSize(const PageSize& o)
  {
    TRACE;
    _paperSize=o._paperSize;
    _unit=o._unit;
    _width=o._width;
    _height= o._height;
    _orientation=o._orientation;
  }

  void PageSize::setPaperSize(QPageSize::PageSizeId id)
  {
    TRACE;
    if(id!=QPageSize::Custom) {
      QPageSize ps(id);
      if(!ps.isValid()) {
        ps=QPageSize(QPageSize::A4);
      }
      _unit=ps.definitionUnits();
      _width=ps.definitionSize().width();
      _height=ps.definitionSize().height();
    }
    _paperSize=id;
  }

  QString PageSize::paperSizeString() const
  {
    TRACE;
    if(_paperSize==QPageSize::Custom) {
      return "Custom";
    } else {
      return "Standard";
    }
  }

  bool PageSize::setPaperSize(QString id)
  {
    TRACE;
    id=id.toLower();
    if(id=="standard") {
      _paperSize=QPageSize::A4; // Anything different from Custom
                                // Exact standard format is computed by
                                // QPageSize from width and height
    } else if(id=="custom"){
      _paperSize=QPageSize::Custom;
    } else {
      return false;
    }
    return true;
  }

  QString PageSize::unitString() const
  {
    TRACE;
    switch(_unit) {
    case QPageSize::Inch:
      return "Inch";
    case QPageSize::Point:
      return "Point";
    default:
      break;
    }
    return "Millimeter";
  }

  bool PageSize::setUnit(QString u)
  {
    TRACE;
    u=u.toLower();
    if(u=="millimeter") {
      _unit=QPageSize::Millimeter;
    } else if(u=="inch") {
      _unit=QPageSize::Inch;
    } else if(u=="point") {
      _unit=QPageSize::Point;
    } else {
      return false;
    }
    return true;
  }

  bool PageSize::setOrientation(QString o)
  {
    TRACE;
    o=o.toLower();
    if(o=="portrait") {
      _orientation=QPageLayout::Portrait;
    } else if(o=="landscape") {
      _orientation=QPageLayout::Landscape;
    } else {
      return false;
    }
    return true;
  }

  QString PageSize::orientationString() const
  {
    switch(_orientation) {
    case QPageLayout::Landscape:
      return "Landscape";
    case QPageLayout::Portrait:
      break;
    }
    return "Portrait";
  }

  void PageSize::properties(PropertyWidget * w) const
  {
    TRACE;
    w->setValue(PageSizeProperties::PaperSize, PageSizeProperties::paperSize2item(_paperSize));
    w->setValue(PageSizeProperties::Unit, PageSizeProperties::unit2item(_unit));
    w->setValue(PageSizeProperties::Orientation, PageSizeProperties::orientation2item(_orientation));

    if(_orientation==QPageLayout::Portrait) {
      w->setValue(PageSizeProperties::Width, _width);
      w->setValue(PageSizeProperties::Height, _height);
    } else {
      w->setValue(PageSizeProperties::Width, _height);
      w->setValue(PageSizeProperties::Height, _width);
    }
  }

  void PageSize::setProperty(int pid, QVariant val)
  {
    switch(pid) {
    case PageSizeProperties::PaperSize:
      setPaperSize(PageSizeProperties::item2paperSize(val.toInt()));
      break;
    case PageSizeProperties::Unit:
      setUnit(PageSizeProperties::item2unit(val.toInt()));
      break;
    case PageSizeProperties::Width:
      if(_orientation==QPageLayout::Portrait) {
        setWidth(val.toDouble());
      } else {
        setHeight(val.toDouble());
      }
      break;
    case PageSizeProperties::Height:
      if(_orientation==QPageLayout::Portrait) {
        setHeight(val.toDouble());
      } else {
        setWidth(val.toDouble());
      }
      break;
    case PageSizeProperties::Orientation:
      setOrientation(PageSizeProperties::item2orientation(val.toInt()));
      break;
    }
  }

  QPageLayout PageSize::pageLayout() const
  {
    TRACE;
    QPageLayout l;
    l.setMode(QPageLayout::FullPageMode);
    l.setOrientation(_orientation);
    if(_paperSize==QPageSize::Custom) {
      l.setPageSize(QPageSize(QSizeF(_width, _height), _unit, QString(), QPageSize::ExactMatch));
    } else {
      l.setPageSize(QPageSize(_paperSize));
    }
    return l;
  }

  void PageSize::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
  {
    TRACE;
    Q_UNUSED(context)
    writeProperty(s, "paperSize", paperSizeString());
    writeProperty(s, "unit", unitString());
    writeProperty(s, "width", width());
    writeProperty(s, "height", height());
    writeProperty(s, "orientation", orientationString());
  }

  XMLMember PageSize::xml_member(XML_MEMBER_ARGS)
  {
    TRACE;
    Q_UNUSED(attributes)
    Q_UNUSED(context)
    if(tag=="paperSize") return XMLMember(0);
    else if(tag=="unit") return XMLMember(1);
    else if(tag=="width") return XMLMember(2);
    else if(tag=="height") return XMLMember(3);
    else if(tag=="orientation") return XMLMember(4);
    else return XMLMember(XMLMember::Unknown);
  }

  bool PageSize::xml_setProperty(XML_SETPROPERTY_ARGS)
  {
    TRACE;
    Q_UNUSED(tag)
    Q_UNUSED(attributes)
    Q_UNUSED(context)
    bool ok=true;
    switch(memberID) {
    case 0:
      return setPaperSize(content.toString());
    case 1:
      return setUnit(content.toString());
    case 2:
      setWidth(content.toDouble(&ok));
      return ok;
    case 3:
      setHeight(content.toDouble(&ok));
      return ok;
    case 4:
      return setOrientation(content.toString());
    default:
      return false;
    }
  }

  bool PageSize::xml_polish(XML_POLISH_ARGS)
  {
    TRACE;
    Q_UNUSED(context)
    if(_paperSize!=QPageSize::Custom) {
      QPageSize ps(QSizeF(_width, _height), _unit, QString(), QPageSize::ExactMatch);
      _paperSize=ps.id();
    }
    return true;
  }


} // namespace SciFigs

