/***************************************************************************
**
**  This file is part of QGpGuiTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2010-06-23
**  Copyright: 2010-2019
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>

#include "MultiDocumentTabBar.h"

namespace QGpGuiTools {

/*!
  \class MultiDocumentTabBar MultiDocumentTabBar.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
MultiDocumentTabBar::MultiDocumentTabBar(QWidget * parent)
    : QTabBar(parent)
{
  TRACE;
  setContextMenuPolicy(Qt::CustomContextMenu);
  connect(this, SIGNAL(customContextMenuRequested(const QPoint&)),
          this, SLOT(showContextMenu(const QPoint&)));

  QAction * a;

  a=new QAction(tr("New tab"), this);
  a->setToolTip(tr("Insert a new tab after this one (or before if it is the first tab)"));
  connect(a, SIGNAL(triggered()), this, SLOT(newTab()));
  addAction(a);

  a=new QAction(tr("Rename"), this);
  a->setToolTip(tr("Rename this tab"));
  connect(a, SIGNAL(triggered()), this, SLOT(renameTab()));
  addAction(a);

  a=new QAction(tr("Close"), this);
  a->setToolTip(tr("Close this tab"));
  connect(a, SIGNAL(triggered()), this, SLOT(closeTab()));
  addAction(a);

  a=new QAction(tr("Tabify"), this);
  a->setToolTip(tr("Move all windows to new tabs"));
  connect(a, SIGNAL(triggered()), this, SLOT(tabify()));
  addAction(a);

  // Code inspired from Hashplat on http://stackoverflow.com
  // http://stackoverflow.com/questions/19975137/how-can-i-add-a-new-tab-button-next-to-the-tabs-of-a-qmdiarea-in-tabbed-view-m
  _plusButton=new QToolButton(this);
  _plusButton->setIcon(QIcon(":newtab.png"));
  _plusButton->setToolButtonStyle(Qt::ToolButtonIconOnly);
  _plusButton->setMinimumSize(QSize(20, 20));
  _plusButton->setMaximumSize(QSize(20, 20));
  _plusButton->setAutoRaise(true);
  _plusButton->setToolTip(tr("Add a new tab"));
  connect(_plusButton, SIGNAL(clicked(bool)), this, SLOT(newTab()));
  movePlusButton();

#if(QT_VERSION >= QT_VERSION_CHECK(5, 4, 0))
  setChangeCurrentOnDrag(true);
#endif
}

/*!
  Description of destructor still missing
*/
MultiDocumentTabBar::~MultiDocumentTabBar()
{
  TRACE;
}

void MultiDocumentTabBar::showContextMenu(const QPoint& pos)
{
  TRACE;
  int index=tabAt(pos);
  if(index>-1) {
    setCurrentIndex(index);
    QMenu m;
    m.addActions(actions());
    QAction * winMenu=nullptr;
    emit windowMenuRequested(index, winMenu);
    QList<QAction *> l=winMenu->menu()->actions();
    if(!l.isEmpty()) {
      m.addSeparator();
      m.addActions(l);
    }
    m.exec(mapToGlobal(pos));
  }
}

void MultiDocumentTabBar::mouseDoubleClickEvent(QMouseEvent * event)
{
  TRACE;
  int index=tabAt(event->pos());
  if(index>-1) {
    emit renameTabRequested(index);
  }
}

void MultiDocumentTabBar::newTab()
{
  TRACE;
  emit newTabRequested(currentIndex());
}

void MultiDocumentTabBar::renameTab()
{
  TRACE;
  emit renameTabRequested(currentIndex());
}

void MultiDocumentTabBar::closeTab()
{
  TRACE;
  emit closeTabRequested(currentIndex());
}

void MultiDocumentTabBar::tabify()
{
  TRACE;
  emit tabifyRequested(currentIndex());
}

/*!
  Return the size of the TabBar with increased width for the plus button.
*/
QSize MultiDocumentTabBar::sizeHint() const
{
  TRACE;
  QSize s=QTabBar::sizeHint();
  return QSize(s.width()+25, s.height());
}

/*!
  Resize the widget and make sure the plus button is in the correct location.
*/
void MultiDocumentTabBar::resizeEvent(QResizeEvent * event)
{
  TRACE;
  QTabBar::resizeEvent(event);
  movePlusButton();
}

/*!
  This virtual handler is called whenever the tab layout changes.
*/
void MultiDocumentTabBar::tabLayoutChange()
{
  TRACE;
  QTabBar::tabLayoutChange();
  movePlusButton();
}

void MultiDocumentTabBar::movePlusButton()
{
  TRACE;
  // Code inspired from Hashplat on http://stackoverflow.com
  // http://stackoverflow.com/questions/19975137/how-can-i-add-a-new-tab-button-next-to-the-tabs-of-a-qmdiarea-in-tabbed-view-m

  // Find the width of all of the tabs
  int size=0;
  for(int i=count()-1; i>=0; i--) {
    size+=tabRect(i).width();
  }

  // Set the plus button location in a visible area
  int h=geometry().top()+(height()-20)/2;
  _plusButton->move(size, h);
}

} // namespace QGpGuiTools
