/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2016-08-23
**  Copyright: 2016-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "RefractionProxy.h"

namespace QGpCoreWave {

  /*!
    \class RefractionProxy RefractionProxy.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  CURVEPROXY_BASICIMPLEMENTATIONS(RefractionProxy)

  QString RefractionProxy::columnName(int col) const
  {
    TRACE;
    switch(col) {
      case 0: return tr("Distance");
      case 1: return tr("Time");
      case 2: return tr("Stddev");
      case 3: return tr("Weight");
      default: return QString();
    }
  }

  QString RefractionProxy::columnUnit(int col) const
  {
    TRACE;
    switch(col) {
      case 0: return tr("m");
      case 1:
      case 2: return tr("s");
      default: return QString();
    }
  }

  QVector<int> RefractionProxy::savedColumns() const
  {
    TRACE;
    QVector<int> l;
    l << 0 << 1 << 2 << 3;
    return l;
  }

  QVariant RefractionProxy::columnValue(int sample, int col) const
  {
    TRACE;
    const RealStatisticalPoint& p=curve().constAt(sample);
    switch(col) {
    case 0: return p.x();
    case 1: return p.mean();
    case 2: return p.stddev();
    case 3: return p.weight();
    default: return QVariant();
    }
  }

  void RefractionProxy::setColumnValue(int sample, int col, const QVariant & value)
  {
    TRACE;
    RealStatisticalPoint& p=curve().constXAt(sample);
    switch(col) {
    case 0: curve().setX(sample, value.toDouble()); break;
    case 1: p.setMean(value.toDouble()); break;
    case 2: p.setStddev(value.toDouble()); break;
    case 3: p.setWeight(value.toDouble()); break;
    default: break;
    }
  }

  QStringList RefractionProxy::columnFileTypes() const
  {
    TRACE;
    static QStringList types;
    if(types.isEmpty()) {
      types << tr("Absolute location (m)")          // 1
            << tr("Relative location (m)")          // 2
            << tr("Time (s)")                       // 3
            << tr("Time stddev (s)")                // 4
            << tr("Weight");                        // 5
    }
    return types;
  }

  QVector<int> RefractionProxy::defaultColumnFileTypes() const
  {
    TRACE;
    QVector<int> types;
    types << 1 << 3 << 4 << 5;
    return types;
  }

  bool RefractionProxy::parse(ColumnTextIterator& it)
  {
    TRACE;
    const ColumnTextParser * parser=it.parser();
    int nColumns=parser->columnCount();
    RefractionCurve& c=curve();
    c.clear();
    bool ok=true;
    bool relativeLocation=false;
    while(!it.atEnd() && c.isEmpty()) {
      while(!it.atSectionEnd()) {
        FactoryPoint p;
        for(int iCol=0;iCol<nColumns;iCol++) {
          switch(parser->type(iCol)) {
          case 1:
            p.setX(parser->text(it.currentRow(), iCol).toDouble(&ok));
            relativeLocation=false;
            break;
          case 2:
            p.setX(parser->text(it.currentRow(), iCol).toDouble(&ok));
            relativeLocation=true;
            break;
          case 3:
            p.setMean(parser->text(it.currentRow(), iCol).toDouble(&ok));
            break;
          case 4:
            p.setStddev(parser->text(it.currentRow(), iCol).toDouble(&ok));
            break;
          case 5:
            p.setWeight(parser->text(it.currentRow(), iCol).toDouble(&ok));
            break;
         default:
            break;
          }
          if(!ok) {
            parser->errorParsingColumn(iCol, it.currentRow());
            return false;
          }
        }
        if(p.mean()==0.0) {
          c.setSource(p.x());
        } else {
          c.append(p);
        }
        it.nextRow();
      }
    }
    if(relativeLocation) {
      for(RefractionCurve::iterator it=c.begin();it!=c.end();it++) {
        RealStatisticalPoint& p=*it;
        p.setX(c.source()+p.x());
      }
    }
    return false;
  }

} // namespace QGpCoreWave

