/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2016-08-23
**  Copyright: 2016-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "RealStatisticalProxy.h"

namespace QGpCoreMath {

  /*!
    \class RealStatisticalProxy RealStatisticalProxy.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  CURVEPROXY_BASICIMPLEMENTATIONS(RealStatisticalProxy)

  QString RealStatisticalProxy::columnName(int col) const
  {
    TRACE;
    switch(col) {
      case 0: return tr("X");
      case 1: return tr("Y");
      case 2: return tr("Stddev");
      case 3: return tr("Weight");
      default: return QString();
    }
  }

  QString RealStatisticalProxy::columnUnit(int) const
  {
    TRACE;
    return QString();
  }

  QVector<int> RealStatisticalProxy::savedColumns() const
  {
    TRACE;
    QVector<int> l;
    l << 0 << 1 << 2 << 3;
    return l;
  }

  QVariant RealStatisticalProxy::columnValue(int sample, int col) const
  {
    TRACE;
    const RealStatisticalPoint& p=curve().constAt(sample);
    switch(col) {
    case 0: return p.x();
    case 1: return p.mean();
    case 2: return p.stddev();
    case 3: return p.weight();
    default: return QVariant();
    }
  }

  void RealStatisticalProxy::setColumnValue(int sample, int col, const QVariant & value)
  {
    TRACE;
    RealStatisticalPoint& p=curve().constXAt(sample);
    switch(col) {
    case 0: curve().setX(sample, value.toDouble()); break;
    case 1: p.setMean(value.toDouble()); break;
    case 2: p.setStddev(value.toDouble()); break;
    case 3: p.setWeight(value.toDouble()); break;
    default: break;
    }
  }

  QStringList RealStatisticalProxy::columnFileTypes() const
  {
    TRACE;
    static QStringList types;
    if(types.isEmpty()) {
      types << tr("X")                  // 1
            << tr("Average")            // 2
            << tr("Stddev")             // 3
            << tr("Weight")             // 4
            << tr("Valid");             // 5
    }
    return types;
  }

  QVector<int> RealStatisticalProxy::defaultColumnFileTypes() const
  {
    TRACE;
    QVector<int> l;
    l << 1 << 2 << 3 << 4 << 5;
    return l;
  }

  bool RealStatisticalProxy::parse(ColumnTextIterator& it)
  {
    TRACE;
    const ColumnTextParser * parser=it.parser();
    int nColumns=parser->columnCount();
    Curve<RealStatisticalPoint>& c=curve();
    c.clear();
    bool ok=true;
    while(!it.atEnd() && c.isEmpty()) {
      while(!it.atSectionEnd()) {
        RealStatisticalPoint p;
        for(int iCol=0;iCol<nColumns;iCol++) {
          switch(parser->type(iCol)) {
          case 1:
            p.setX(parser->text(it.currentRow(), iCol).toDouble(&ok));
            break;
          case 2:
            p.setMean(parser->text(it.currentRow(), iCol).toDouble(&ok));
            break;
          case 3:
            p.setStddev(parser->text(it.currentRow(), iCol).toDouble(&ok));
            break;
          case 4:
            p.setWeight(parser->text(it.currentRow(), iCol).toDouble(&ok));
            break;
          case 5:
            p.setValid(parser->text(it.currentRow(), iCol).toInt(&ok)==1);
            break;
         default:
            break;
          }
          if(!ok) {
            parser->errorParsingColumn(iCol, it.currentRow());
            return false;
          }
        }
        c.append(p);
        it.nextRow();
      }
    }
    return true;
  }

  void RealStatisticalProxy::beginSplit(double maxX, SamplingOptions options, double maxErr, int minCount)
  {
    ASSERT(!_split);
    _split=new QList<Curve<RealStatisticalPoint>>;
    *_split=curve().split(maxX, options, maxErr, minCount);
  }

  int RealStatisticalProxy::splitCount() const
  {
    ASSERT(_split);
    return _split->count();
  }

  void RealStatisticalProxy::getSplit(const CurveProxy * o, int index)
  {
    QList<Curve<RealStatisticalPoint>> * split;
    split=static_cast<const RealStatisticalProxy *>(o)->_split;
    ASSERT(split);
    curve()=split->at(index);
  }

  void RealStatisticalProxy::endSplit()
  {
    delete _split;
    _split=nullptr;
  }

} // namespace QGpCoreMath

