/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-04-01
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "PrimeFactorization.h"

namespace QGpCoreMath {

  /*!
    \class PrimeFactorization PrimeFactorization.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*
     The last number of the table must larger than PRIMEFACTORIZATION_MAXIMUM_PRIME
     With the maximum prime as 1999, if a value is less than 3996001, the result is exact.
     If it is larger, the returned list may be incomplete.

     PRIMEFACTORIZATION_PRIME_COUNT must be also updated. The last number must not be counted.
  */
  #if PRIMEFACTORIZATION_MAXIMUM_PRIME!=1999
  #  error
  #endif
  const int PrimeFactorization::primes[]={       2,      3,      5,      7,     11,     13,     17,     19,     23,     29,
                                                31,     37,     41,     43,     47,     53,     59,     61,     67,     71,
                                                73,     79,     83,     89,     97,    101,    103,    107,    109,    113,
                                               127,    131,    137,    139,    149,    151,    157,    163,    167,    173,
                                               179,    181,    191,    193,    197,    199,    211,    223,    227,    229,
                                               233,    239,    241,    251,    257,    263,    269,    271,    277,    281,
                                               283,    293,    307,    311,    313,    317,    331,    337,    347,    349,
                                               353,    359,    367,    373,    379,    383,    389,    397,    401,    409,
                                               419,    421,    431,    433,    439,    443,    449,    457,    461,    463,
                                               467,    479,    487,    491,    499,    503,    509,    521,    523,    541,
                                               547,    557,    563,    569,    571,    577,    587,    593,    599,    601,
                                               607,    613,    617,    619,    631,    641,    643,    647,    653,    659,
                                               661,    673,    677,    683,    691,    701,    709,    719,    727,    733,
                                               739,    743,    751,    757,    761,    769,    773,    787,    797,    809,
                                               811,    821,    823,    827,    829,    839,    853,    857,    859,    863,
                                               877,    881,    883,    887,    907,    911,    919,    929,    937,    941,
                                               947,    953,    967,    971,    977,    983,    991,    997,   1009,   1013,
                                              1019,   1021,   1031,   1033,   1039,   1049,   1051,   1061,   1063,   1069,
                                              1087,   1091,   1093,   1097,   1103,   1109,   1117,   1123,   1129,   1151,
                                              1153,   1163,   1171,   1181,   1187,   1193,   1201,   1213,   1217,   1223,
                                              1229,   1231,   1237,   1249,   1259,   1277,   1279,   1283,   1289,   1291,
                                              1297,   1301,   1303,   1307,   1319,   1321,   1327,   1361,   1367,   1373,
                                              1381,   1399,   1409,   1423,   1427,   1429,   1433,   1439,   1447,   1451,
                                              1453,   1459,   1471,   1481,   1483,   1487,   1489,   1493,   1499,   1511,
                                              1523,   1531,   1543,   1549,   1553,   1559,   1567,   1571,   1579,   1583,
                                              1597,   1601,   1607,   1609,   1613,   1619,   1621,   1627,   1637,   1657,
                                              1663,   1667,   1669,   1693,   1697,   1699,   1709,   1721,   1723,   1733,
                                              1741,   1747,   1753,   1759,   1777,   1783,   1787,   1789,   1801,   1811,
                                              1823,   1831,   1847,   1861,   1867,   1871,   1873,   1877,   1879,   1889,
                                              1901,   1907,   1913,   1931,   1933,   1949,   1951,   1973,   1979,   1987,
                                              1993,   1997,   1999,   2003};

  void PrimeFactorization::setValue(int val)
  {
    int i=0;
    Factor f;
    f.base=primes[i++];
    f.exponent=0;
    // Run base==2 separately as it can avoid divisions
    while(f.base<val) {
      if(val & 1) {
        if(f.exponent>0) {
          _factors.append(f);
          f.exponent=0;
        }
        f.base=primes[i++];
        if(f.base>PRIMEFACTORIZATION_MAXIMUM_PRIME) {
          _valid=false;
          return;
        }
        break;
      } else {
        val=val >> 1;
        f.exponent++;
      }
    }
    // Run for other bases
    int q=val;
    while(f.base<q) {
      q=val/f.base;
      if(q*f.base==val) {
        val=q;
        f.exponent++;
      } else {
        if(f.exponent>0) {
          _factors.append(f);
          f.exponent=0;
        }
        f.base=primes[i++];
        if(f.base>PRIMEFACTORIZATION_MAXIMUM_PRIME) {
          if(primes[i-1]<q) {
            _valid=false;
            return;
          } else {
            break;
          }
        }
      }
    }
    if(val==f.base) {
      f.exponent++;
    } else {
      f.base=val;
      f.exponent=1;
    }
    _factors.append(f);
    _valid=true;
  }

  int PrimeFactorization::value() const
  {
    int val=1;
    for(int i=_factors.count()-1; i>=0; i--) {
      const Factor& f=_factors[i];
      for(int j=f.exponent; j>0; j--) {
        val*=f.base;
      }
    }
    return val;
  }

  bool PrimeFactorization::isSquared() const
  {
    for(int i=_factors.count()-1; i>=0; i--) {
      const Factor& f=_factors[i];
      if(f.exponent & 1) {
        return false;
      }
    }
    return true;
  }

  /*!
    Return the square root rounded to the lower exact integer root.
  */
  void PrimeFactorization::sqrtLow()
  {
    for(int i=_factors.count()-1; i>=0; i--) {
      Factor& f=_factors[i];
      if(f.exponent>1) {
        f.exponent/=2;
      } else {
        _factors.removeAt(i);
      }
    }
  }

  /*!
    Return the square root rounded to the higher exact integer root.
  */
  void PrimeFactorization::sqrtHigh()
  {
    for(int i=_factors.count()-1; i>=0; i--) {
      Factor& f=_factors[i];
      f.exponent=(f.exponent+1)/2;
    }
  }

  /*!
    Not a true division. Only the common factors are divided.
    To be used with
  */
  void PrimeFactorization::operator/=(const PrimeFactorization& o)
  {
    int i=_factors.count()-1;
    int j=o._factors.count()-1;
    while(i>=0 && j>=0) {
      Factor& f1=_factors[i];
      const Factor& f2=o._factors[j];
      if(f1.base>f2.base) {
        do {
          i--;
        } while (i>=0 && _factors[i].base>f2.base);
      } else if(f1.base<f2.base) {
        do {
          j--;
        } while (j>=0 && o._factors[j].base>f1.base);
      } else {
        f1.exponent=-f2.exponent;
        if(f1.exponent<1) {
          _factors.removeAt(i);
        }
      }
    }
  }

  QString PrimeFactorization::toString() const
  {
    QString str;
    if(_factors.count()>0) {
      const Factor& f=_factors[0];
      if(f.exponent==1) {
        str=QString::number(f.base);
      } else {
        str=QString::number(f.base)+"^"+QString::number(f.exponent);
      }
      for(int i=1; i<_factors.count(); i++) {
        const Factor& f=_factors[i];
        if(f.exponent==1) {
          str+="*"+QString::number(f.base);
        } else {
          str+="*"+QString::number(f.base)+"^"+QString::number(f.exponent);
        }
      }
    }
    return str;
  }
  /*!
    Returns true if a prime factorization of \a n is possible
    with the maximum prime number equal to \a maxPrime.
  */
  bool PrimeFactorization::isValid(int val, int maxPrime)
  {
    int n0=val;
    if(maxPrime>PRIMEFACTORIZATION_MAXIMUM_PRIME) {
      maxPrime=PRIMEFACTORIZATION_MAXIMUM_PRIME;
    }
    int i=0;
    int base=primes[i++];
    int factorization=1;
    // Run base==2 separately as it can avoid divisions
    while(base<val) {
      if(val & 1) {
        base=primes[i++];
        if(base>maxPrime) {
          return false;
        }
        break;
      } else {
        val=val >> 1;
        factorization*=base;
      }
    }
    // Run for other bases
    int q=val;
    while(base<q) {
      q=val/base;
      if(q*base==val) {
        val=q;
        factorization*=base;
      } else {
        base=primes[i++];
        if(base>maxPrime) {
          if(primes[i-1]<q) {
            return false;
          } else {
            break;
          }
        }
      }
    }
    factorization*=val;
    return factorization==n0 && val<=maxPrime;
  }

  /*!
    Used to generate the list of prime number for
  */
  void PrimeFactorization::primeList(int min, int max)
  {
    PrimeFactorization f;
    for(int i=min; i<=max; i++) {
      f.setValue(i);
      if(f.isPrime()) {
        printf(", %6i", i);
      }
    }
  }

  int PrimeFactorization::closestValue(int n, int maxPrime)
  {
    if(isValid(n, maxPrime)) {
      return n;
    }
    int nPlus=n+1;
    int nLess=n-1;
    while(true) {
      if(isValid(nPlus, maxPrime)) {
        return nPlus;
      }
      if(isValid(nLess, maxPrime)) {
        return nLess;
      }
      nPlus++;
      nLess--;
    }
  }

  /*!
    Run consistency tests on the the prime table and on the factorization.
  */
  bool PrimeFactorization::test()
  {
    if(primes[PRIMEFACTORIZATION_PRIME_COUNT-1]!=PRIMEFACTORIZATION_MAXIMUM_PRIME) {
      App::log(tr("PRIMEFACTORIZATION_PRIME_COUNT and PRIMEFACTORIZATION_MAXIMUM_PRIME are not correct.\n"
                  "primes[PRIMEFACTORIZATION_PRIME_COUNT-1]=%i\n")
               .arg(primes[PRIMEFACTORIZATION_PRIME_COUNT-1]));
      return false;
    }
    if(primes[PRIMEFACTORIZATION_PRIME_COUNT]<=PRIMEFACTORIZATION_MAXIMUM_PRIME) {
      App::log(tr("The last element of prime table is not larger than PRIMEFACTORIZATION_MAXIMUM_PRIME.\n"
                  "primes[PRIMEFACTORIZATION_PRIME_COUNT]=%i\n")
               .arg(primes[PRIMEFACTORIZATION_PRIME_COUNT]));
      return false;
    }

    App::log(tr("Checking basic prime numbers...\n"));
    for(int i=0; i<PRIMEFACTORIZATION_PRIME_COUNT; i++) {
      PrimeFactorization pf(primes[i]);
      if(!pf.isPrime()) {
        App::log(tr("%1 should be detected as a prime number\n").arg(primes[i]));
        return false;
      }
    }

    int n=PRIMEFACTORIZATION_MAXIMUM_PRIME*PRIMEFACTORIZATION_MAXIMUM_PRIME;
    App::log(tr("Checking prime factorization up to %1...\n").arg(n));
    for(int i=0; i<n; i++) {
      PrimeFactorization pf(i);
      for(int j=pf.count()-1; j>=0; j--) {
        if(pf.factor(j).exponent<=0) {
          App::log(tr("Null or negative exponent for %1\n").arg(i));
          return false;
        }
      }
      if(!pf.isValid()) {
        App::log(tr("Prime factorization is not valid for %1=%2\n")
                 .arg(i)
                 .arg(pf.toString()));
        return false;
      }
      if(i!=pf.value()) {
        App::log(tr("Product of prime factors is not equal to %1\n").arg(i));
        return false;
      }
    }
    App::log(tr("Checking static function for prime factorization up to %1...\n").arg(n));
    for(int i=0; i<PRIMEFACTORIZATION_MAXIMUM_PRIME; i++) {
      if(!isValid(i, PRIMEFACTORIZATION_MAXIMUM_PRIME)) {
        PrimeFactorization pf(i);
        App::log(tr("Prime factorization is not valid as returned by the static function for %1=%2\n")
                 .arg(i).arg(pf.toString()));
        return false;
      }
    }

    for(int imp=0; imp<20; imp++) {
      int maxPrime=primes[imp];
      App::log(tr("Checking closest value with maximum prime to %1...\n").arg(maxPrime));
      for(int i=0; i<100000; i++) {
        int ic=closestValue(i, maxPrime);
        PrimeFactorization pfic(ic);
        for(int j=pfic.count()-1; j>=0; j--) {
          if(pfic.factor(j).base>maxPrime) {
            PrimeFactorization pfi(i);
            App::log(tr("Closest value is not a prime factorization with a maximum of %5\n"
                        "%1=%2 --> %3=%4\n")
                     .arg(i)
                     .arg(pfi.toString())
                     .arg(ic)
                     .arg(pfic.toString())
                     .arg(maxPrime));
            return false;
          }
        }
        if(!isValid(ic, maxPrime)) {
          PrimeFactorization pfi(i);
          App::log(tr("Prime factorization is not valid as returned by the static function\n"
                      "%1=%2 --> %3=%4\n")
                   .arg(i)
                   .arg(pfi.toString())
                   .arg(ic)
                   .arg(pfic.toString()));
          return false;
        }
      }
    }
    return true;
  }

} // namespace QGpCoreMath

