/***************************************************************************
**
**  This file is part of HVGui.
**
**  HVGui is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  HVGui is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-09-23
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpGuiTools.h>
#include <SciFigs.h>
#include "SummaryMapOptions.h"

namespace HVGui {

/*!
  \class SummaryMapOptions SummaryMapOptions.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
SummaryMapOptions::SummaryMapOptions(QWidget * parent)
    : QWidget(parent)
{
  TRACE;
  setupUi(this);
  _legendProperties=0;
  connect(peakNumber, SIGNAL(clicked()), this, SLOT(commit()));
  connect(peakIndexEdit, SIGNAL(valueChanged(int)), this, SLOT(commit()));
  connect(lowestPeak, SIGNAL(clicked()), this, SLOT(commit()));
  connect(highestPeak, SIGNAL(clicked()), this, SLOT(commit()));
  connect(maxAmplitudePeak, SIGNAL(clicked()), this, SLOT(commit()));
  connect(peakFrequency, SIGNAL(clicked()), this, SLOT(commit()));
  connect(peakAmplitude, SIGNAL(clicked()), this, SLOT(commit()));
  connect(labelStationName, SIGNAL(clicked()), this, SLOT(commit()));
  connect(labelPeakValue, SIGNAL(clicked()), this, SLOT(commit()));
  connect(peakValueDecimalsEdit, SIGNAL(valueChanged(int)), this, SLOT(commit()));
  connect(noLabel, SIGNAL(clicked()), this, SLOT(commit()));
  connect(showBubblesEdit, SIGNAL(clicked()), this, SLOT(commit()));
}

/*!
  Description of destructor still missing
*/
SummaryMapOptions::~SummaryMapOptions()
{
  TRACE;
}

void SummaryMapOptions::on_peakNumber_toggled(bool checked)
{
  peakIndexEdit->setEnabled(checked);
}

void SummaryMapOptions::on_labelPeakValue_toggled(bool checked)
{
  peakValueDecimalsLabel->setEnabled(checked);
  peakValueDecimalsEdit->setEnabled(checked);
}

void SummaryMapOptions::on_categoryLegend_clicked()
{
  TRACE;
  TRACE_BUG;
  Dialog * d=new Dialog(this);
  ASSERT(!_legendProperties);
  _legendProperties=new LegendProperties(this);
  _legendProperties->setPropertySections(LegendTable::TextAll | LegendTable::SymbolAll);
  TRACE_BUG_N(1);
  d->setMainWidget(_legendProperties, Dialog::TitleClose);
  TRACE_BUG_N(2);
  _legendProperties->setLegend(_categories);
  _legendProperties->setReadOnlyText(true);
  TRACE_BUG_N(3);
  Settings::getRect(d, "SummaryMapOptions::legend");
#ifndef Q_OS_MAC
  connect(_legendProperties, SIGNAL(touched()), this, SLOT(commit()));
#endif
  TRACE_BUG_N(4);
  d->exec();
#ifdef Q_OS_MAC
  commit();
#endif
  Settings::setRect(d, "SummaryMapOptions::legend");
  TRACE_BUG_N(5);
  delete d;
  TRACE_BUG_N(6);
  _legendProperties=0;
}

void SummaryMapOptions::commit()
{
  TRACE;
  if(_legendProperties) {
    _categories=_legendProperties->legend();
  }
  emit touched();
}

void SummaryMapOptions::setLabels(StationLine::Label l)
{
  TRACE;
  switch(l) {
  case StationLine::StationName:
    labelStationName->setChecked(true);
    break;
  case StationLine::PeakValue:
    labelPeakValue->setChecked(true);
    break;
  case StationLine::NoLabel:
    noLabel->setChecked(true);
    break;
  }
}

void SummaryMapOptions::setPeakSelection(StationLine::PeakSelection v)
{
  switch(v) {
  case StationLine::PeakNumber:
    peakNumber->setChecked(true);
    break;
  case StationLine::PeakLowest:
    lowestPeak->setChecked(true);
    break;
  case StationLine::PeakHighest:
    highestPeak->setChecked(true);
    break;
  case StationLine::PeakMaximum:
    maxAmplitudePeak->setChecked(true);
    break;
  }
}

void SummaryMapOptions::setPeakValue(CurveResults::PeakValue v)
{
  switch(v) {
  case CurveResults::PeakFrequency:
    peakFrequency->setChecked(true);
    break;
  case CurveResults::PeakAmplitude:
    peakAmplitude->setChecked(true);
    break;
  }
}

void SummaryMapOptions::setPeakIndex(int index)
{
  peakIndexEdit->setValue(index+1);
}

void SummaryMapOptions::setShowBubbles(bool b)
{
  showBubblesEdit->setChecked(b);
}

void SummaryMapOptions::setPeakValueDecimals(int d)
{
  peakValueDecimalsEdit->setValue(d);
}

StationLine::Label SummaryMapOptions::labels() const
{
  if(labelPeakValue->isChecked()) {
    return StationLine::PeakValue;
  } else if(noLabel->isChecked()) {
    return StationLine::NoLabel;
  } else {
    return StationLine::StationName;
  }
}

StationLine::PeakSelection SummaryMapOptions::peakSelection() const
{
  if(lowestPeak->isChecked()) {
    return StationLine::PeakLowest;
  } else if(highestPeak->isChecked()) {
    return StationLine::PeakHighest;
  } else if(maxAmplitudePeak->isChecked()) {
    return StationLine::PeakMaximum;
  } else {
    return StationLine::PeakNumber;
  }
}

CurveResults::PeakValue SummaryMapOptions::peakValue() const
{
  if(peakFrequency->isChecked()) {
    return CurveResults::PeakFrequency;
  } else {
    return CurveResults::PeakAmplitude;
  }
}

int SummaryMapOptions::peakIndex() const
{
  return peakIndexEdit->value()-1;
}

bool SummaryMapOptions::showBubbles() const
{
  return showBubblesEdit->isChecked();
}

int SummaryMapOptions::peakValueDecimals() const
{
  return peakValueDecimalsEdit->value();
}

} // namespace HVGui
