/***************************************************************************
**
**  This file is part of HVCore.
**
**  HVCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  HVCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2020-02-20
**  Copyright: 2020
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HVStation.h"
#include "CurveResults.h"
#include "HVStationSignals.h"

namespace HVCore {

  /*!
    \class HVStation HVStation.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  HVStation::HVStation(StationSignals * sig)
    : AbstractStation(new StationProcessSignals(sig))
  {
    TRACE;
  }

  bool HVStation::save(int, QDir outputDir, const HVParameters& param)
  {
    TRACE;
    QString fileName=_signals->originalSignals()->name();;
    fileName=outputDir.absoluteFilePath(fileName+".hv");
    return _results[0]->save(fileName, log(param));
  }

  StationProcessSignals * HVStation::createProcessSignals()
  {
    return new HVStationSignals;
  }

  AbstractResults * HVStation::createResults()
  {
    AbstractResults * r=new CurveResults(this);
    r->setName(originalSignals()->name());
    connect(r, SIGNAL(colorsToChange(const QVector<int>*, const Color&)),
            this, SLOT(changeColors(const QVector<int>*, const Color&)));
    return r;
  }

  void HVStation::process(StationProcessSignals * proc, int windowIndex, const HVParameters * param) const
  {
    TRACE;
    HVStationSignals * hvProc=static_cast<HVStationSignals *>(proc);
    const TimeWindow& r=windowAt(windowIndex);
    hvProc->setProcessed(_signals, r, param);

    Curve<Point2D> hv;
    switch (param->horizontal()) {
    case HVParameters::Energy:
      hv=hvProc->horizontal(param);
      break;
    case HVParameters::Azimuth:
      hv=hvProc->horizontal(param->horizontalAzimuth(), param);
      break;
    case HVParameters::Squared:
      hv=hvProc->horizontal(param, 0.5);
      break;
    case HVParameters::Geometric:
      hv=hvProc->horizontalGeometric(param);
      break;
    }
    hv.yDivide(hvProc->vertical());
    _results[0]->setWindow(windowIndex, r.lengthSeconds(), hv);
    _results[0]->setWindowColor(windowIndex, r.color());
  }


  /*!
    Iterative frequency domain window rejection according to Cox, Cheng, Vantassel and Manuel (2020) in GJI.
  */
  bool HVStation::frequencyWindowRejection(const FrequencyWindowRejection& param)
  {
    CurveResults * results=static_cast<CurveResults *>(_results[0]);
    int n=results->windowCount();
    if(n==0) {
      App::log(tr("No window available, compute H/V first"));
      return false;
    }
    if(n!=windows().count()) {
      App::log(tr("Results are not up-to-date. Run H/V first"));
      return false;
    }
    // Init with all available time windows
    QVector<bool> selected(n);
    QVector<double> f0(n);
    for(int i=0; i<n; i++) {
      f0[i]=results->peak(i, param.minimumFrequency(), param.maximumFrequency());
      selected[i]=(f0[i]>0.0);
    }
    Statistics s;
    s.reset();
    for(int i=0; i<n; i++) {
      if(selected.at(i)) {
        s.addLog(f0.at(i));
      }
    }
    double lmf0=s.meanLog();
    double sigmab;
    double sigmae=s.stddevLog();
    double f0mc=results->averagePeak(selected, param.minimumFrequency(), param.maximumFrequency());
    double db;
    double de=fabs(lmf0-f0mc);
    int iteration=0;
    do {
      iteration++;
      if(App::verbosity()>=1) {
        int nWin=0;
        for(int i=selected.count()-1; i>=0; i--) {
          if(selected.at(i)) {
            nWin++;
          }
        }
        App::log(tr("Station '%1': iteration %2, %3 windows\n").arg(name()).arg(iteration).arg(nWin));
      }
      db=de;
      sigmab=sigmae;
      double range=::pow(sigmab, param.stddevFactor());
      double minf0=lmf0/range;
      double maxf0=lmf0*range;
      APP_LOG(2, tr("  f0 from average= %1\n"
                    "  f0 from windows= %2 %3 %4\n").arg(f0mc).arg(minf0).arg(lmf0).arg(maxf0));
      for(int i=selected.count()-1; i>=0; i--) {
        double f=f0.at(i);
        if(f<minf0 || f>maxf0) {
          selected[i]=false;
        }
      }
      s.reset();
      for(int i=selected.count()-1; i>=0; i--) {
        if(selected.at(i)) {
          s.addLog(f0.at(i));
        }
      }
      if(s.count()==0) {
        break;
      }
      lmf0=s.meanLog();
      sigmae=s.stddevLog();
      f0mc=results->averagePeak(selected, param.minimumFrequency(), param.maximumFrequency());
      de=fabs(lmf0-f0mc);
      APP_LOG(2, tr("  fabs(de-db)/db=              %1\n"
                    "  fabs(ln(sigmae)-ln(sigmab))= %2\n").arg(fabs(de-db)/db).arg(fabs(::log(sigmae)-::log(sigmab))));
      if(iteration>param.maximumIterationCount()) {
        App::log(tr("Frequency window rejection: maximum number of iteration reached.\n"));
        break;
      }
    } while(fabs(de-db)/db>0.01 || fabs(::log(sigmae)-::log(sigmab))>0.01);

    // Remove all windows that are not selected
    for(int i=n-1; i>=0; i--) {
      if(!selected.at(i)) {
        windows().remove(i);
      }
    }
    return true;
  }

} // namespace HVCore

