/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-05-15
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>

#include "FilterParameters.h"

namespace GeopsyCore {

  /*!
    \class FilterParameters FilterParameters.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  FilterParameters::FilterParameters()
  {
    _fmin=1.0;
    _fmax=std::numeric_limits<double>::quiet_NaN();;
    _band=LowPass;
    _method=FrequencyWindow;
    _window.setShape(WindowFunctionParameters::Cosine);
    _width=0.1;
  }

  FilterParameters::FilterParameters(const FilterParameters &o)
    : AbstractParameters(o),
      _window(o._window)
  {
    _band=o._band;
    setFrequencyRange(o._fmin, o._fmax);
    _method=o._method;
    _order=o._order;
    _width=o._width;
  }

  void FilterParameters::setBand(Band b)
  {
    _band=b;
    setFrequencyRange(_fmin, _fmax);
    _window.setReversed(_band==BandReject);
  }

  void FilterParameters::setFrequencyRange(double min, double max)
  {
    TRACE;
    switch(_band) {
    case LowPass:
    case HighPass:
      _fmin=min;
      _fmax=std::numeric_limits<double>::quiet_NaN();
      break;
    case BandPass:
    case BandReject:
      if(min<max) {
        _fmin=min;
        _fmax=max;
      } else {
        _fmin=max;
        _fmax=min;
      }
      break;
    }
  }

  QString FilterParameters::bandString() const
  {
    switch(_band) {
    case LowPass:
      break;
    case HighPass:
      return "HighPass";
    case BandPass:
      return "BandPass";
    case BandReject:
      return "BandReject";
    }
    return "LowPass";
  }

  void FilterParameters::setBand(const QString& b)
  {
    if(!b.isEmpty()) {
      QString bl=b.toLower();
      switch(bl[0].unicode()) {
      case 'l':
        setBand(LowPass);
        break;
      case 'h':
        setBand(HighPass);
        break;
      case 'b':
        if(bl=="bandpass" ||
           bl=="band pass") {            // For compatibility
          setBand(BandPass);
        } else if(bl=="bandreject" ||
                  bl=="band reject") {   // For compatibility
          setBand(BandReject);
        }
        break;
      default:
        break;
      }
    }
  }

  QString FilterParameters::methodString() const
  {
    switch(_method) {
    case Butterworth:
      break;
    case ChebyshevTypeI:
      return "ChebyshevTypeI";
    case FrequencyWindow:
      return "FrequencyWindow";
    }
    return "Butterworth";
  }

  void FilterParameters::setMethod(const QString& m)
  {
    if(!m.isEmpty()) {
      QString ml=m.toLower();
      switch(ml[0].unicode()) {
      case 'b':
        if(ml=="butterworth") {
          _method=Butterworth;
          return;
        }
        break;
      case 'f':
        if(ml=="frequencywindow") {
          _method=FrequencyWindow;
          return;
        }
      case 'c':
        if(ml=="chebyshevtypei") {
          _method=ChebyshevTypeI;
        } else if(ml=="convolution" ||  // For compatibility
                  ml=="cosineTaper") {  // For compatibility
          _method=FrequencyWindow;
          return;
        }
        break;
      default:
        break;
      }
    }
    App::log(tr("Bad filter method '%1'.\n").arg(m) );
  }

  void FilterParameters::setVersion(PARAMETERS_SETVERSION_ARGS)
  {
    TRACE;
    _window.setVersion(version);
    AbstractParameters::setVersion(version);
  }

  int FilterParameters::keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const
  {
    return 7+AbstractParameters::keywordCount();
  }

  int FilterParameters::totalKeywordCount(PARAMETERS_TOTALKEYWORDCOUNT_ARGS) const
  {
    return AbstractParameters::totalKeywordCount()
        +_window.totalKeywordCount();
  }

  void FilterParameters::collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS)
  {
    TRACE;
    int baseIndex=AbstractParameters::keywordCount();
    keywords.add(prefix+"FILTER_TYPE"+suffix, this, baseIndex);
    keywords.add(prefix+"FILTER_METHOD"+suffix, this, baseIndex+1);
    keywords.add(prefix+"FILTER_MIN_FREQUENCY"+suffix, this, baseIndex+2);
    keywords.add(prefix+"FILTER_MAX_FREQUENCY"+suffix, this, baseIndex+3);
    keywords.add(prefix+"FILTER_CAUSAL"+suffix, this, baseIndex+4);
    keywords.add(prefix+"FILTER_ORDER"+suffix, this, baseIndex+5);
    keywords.add(prefix+"FILTER_WIDTH"+suffix, this, baseIndex+6);
    _window.collectKeywords(keywords, prefix+"FILTER_", suffix);
  }

  bool FilterParameters::setValue(PARAMETERS_SETVALUE_ARGS)
  {
    TRACE;
    switch(index-AbstractParameters::keywordCount()) {
    case 0:
      setBand(value);
      return true;
    case 1:
      setMethod(value);
      return true;
    case 2:
      _fmin=value.toDouble();
      return true;
    case 3:
      _fmax=value.toDouble();
      return true;
    case 4:
      setCausal(value=="y");
      return true;
    case 5:
      setOrder(value.toInt());
      return true;
    default:
      break;
    }
    return AbstractParameters::setValue(index, value, unit, keywords);
  }

  QString FilterParameters::toString(PARAMETERS_TOSTRING_ARGS_IMPL) const
  {
    TRACE;
    QString log;
    log+=prefix+"FILTER_TYPE"+suffix+"="+bandString()+"\n";
    log+=prefix+"FILTER_METHOD"+suffix+"="+methodString()+"\n";
    log+=prefix+"FILTER_MIN_FREQUENCY"+suffix+" (Hz)="+QString::number(_fmin)+"\n";
    log+=prefix+"FILTER_MAX_FREQUENCY"+suffix+" (Hz)="+QString::number(_fmax)+"\n";
    switch(_method) {
    case Butterworth:
      log+=prefix+"FILTER_CAUSAL"+suffix+" (y/n)="+(causal() ? "y" : "n")+"\n";
      log+=prefix+"FILTER_ORDER"+suffix+"="+QString::number(_order)+"\n";
      break;
    case ChebyshevTypeI:
      log+=prefix+"FILTER_ORDER"+suffix+"="+QString::number(_order)+"\n";
      break;
    case FrequencyWindow:
      log+=_window.toString(prefix+"FILTER_", suffix);
      log+=prefix+"FILTER_WIDTH"+suffix+"="+QString::number(_width);
      break;
    }
    return log;
  }

  /*!
    Minimum and maximum frequencies, and width must be already set correcty.

    \a passBandRippledB is the allowable passband ripple measured in decibels.
    \a stopBandAttenuationdB is the minimum attenuation in the stop band, also in decibels.

    Inspired from Octave signal toolbox.
  */
  void FilterParameters::setChebyshevOrder(double passBandRippledB, double stopBandAttenuationdB)
  {
    TRACE;
    // prewarping https://en.wikipedia.org/wiki/Bilinear_transform#Frequency_warping
    //Wpw = (2 / T) .* tan (M_PI .* Wp ./ T);
    //Wsw = (2 / T) .* tan (M_PI .* Ws ./ T);
    double wp, ws;
    switch(_band) {
    case LowPass:
      wp=_fmin*(1.0-_width);
      ws=_fmin;
      break;
    case HighPass:
      ws=_fmin*(1.0+_width);
      wp=_fmin;
      break;
    case BandPass:
      wp=_fmin;
      break;
    case BandReject:
      break;
    }
    /*
      The stop band attenuation (Rs) must be effective at frequency ws
      The pass band ripple attenuation (Rp) must be effective at frequency wp

      Filter definition: G=1/sqrt(1+epsilon^2*T_n^2(w/w0))
      where w0=wp

      Stop band attenuation:
      10^(-Rs/20)=1/sqrt(1+epsilon^2*T_n^2(wp/ws)
      sqrt(10^(Rs/10)-1)/epsilon=T_n(ws/wp)

      By definition,
      T_n(cosh(x))=cosh(n*x)

      T_n(ws/wp)=T_n(ws/wp)=T_n(cosh(acosh(ws/wp)))=cosh(n*acosh(ws/wp))

      epsilon=1/sqrt(10^(-Rp/10)-1), because G=1/sqrt(1+epsilon^2) at wp

      Hence,

      sqrt((10^(Rs/10)-1)/(10^(-Rp/10)-1))=cosh(n*acosh(ws/wp))
      acosh(sqrt((10^(Rs/10)-1)/(10^(-Rp/10)-1)))=n*acosh(ws/wp)
      And we get the value of n
    */
    // Conversions from dB scale
    double stopAttenuation=pow(10.0, 0.1*stopBandAttenuationdB);
    double passAttenuation=pow(10.0, 0.1*passBandRippledB);
    _order=qCeil(acosh(sqrt((stopAttenuation-1.0)/(passAttenuation-1.0)))/acosh(ws/wp));
    _rippleFactor=1/sqrt(passAttenuation-1.0);
  }

} // namespace GeopsyCore
