/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-06-15
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "ComplexSignal.h"
#include "DoubleSignal.h"

namespace GeopsyCore {

/*!
  \class ComplexSignal ComplexSignal.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Takes the square root of all samples
*/
bool ComplexSignal::sqrt()
{
  TRACE;
  LOCK_SAMPLES(Complex, thisSamples, this)
    for(int i=0; i<_nSamples;i++)
      thisSamples[i]=QGpCoreTools::sqrt(thisSamples[i]);
  UNLOCK_SAMPLES(this) else return false;
  return true;
}

/*!
  Multiplies all samples by \a d
*/
bool ComplexSignal::multiply(double d)
{
  TRACE;
  LOCK_SAMPLES(Complex, thisSamples, this)
    for(int i=0; i<_nSamples; i++)
      thisSamples[i]*=d;
  UNLOCK_SAMPLES(this) else return false;
  return true;
}

/*!
  Takes the square of all samples
*/
bool ComplexSignal::square()
{
  TRACE;
  LOCK_SAMPLES(Complex, thisSamples, this)
    for(int i=0; i<_nSamples; i++)
      thisSamples[i]*=thisSamples[i];
  UNLOCK_SAMPLES(this) else return false;
  return true;
}

/*!
  Retrun a curve with the absolute value.
  \a df is the sampling step in frequency domain.
*/
Curve<Point2D> ComplexSignal::abs(double df) const
{
  TRACE;
  Curve<Point2D> c(_nSamples);
  CONST_LOCK_SAMPLES(Complex, thisSamples, this)
    for(int i=0; i<_nSamples; i++) {
      Point2D& p=c.constXAt(i);  // not true but sort afterwards
      p.setX(static_cast<double>(i)*df);
      p.setY(thisSamples[i].abs());
    }
  UNLOCK_SAMPLES(this)
  c.sort();
  return c;
}

/*!
  Retrun a double signal with complex numbers arranged according to real Fourier transform
*/
DoubleSignal * ComplexSignal::toDouble(DoubleSignal * sig) const
{
  TRACE;
  if(!sig) {
    sig=new DoubleSignal(_nSamples);
  } else {
    sig->setNSamples(_nSamples);
  }
  CONST_LOCK_SAMPLES(Complex, thisSamples, this)
    LOCK_SAMPLES(double, sigSamples, sig)
      for(int i=0; i<_nSamples; i++) {
        sigSamples[i]=thisSamples[i].re();
        sigSamples[2*i]=thisSamples[i].re();
      }
    UNLOCK_SAMPLES(sig)
  UNLOCK_SAMPLES(this)
  sig->setType(DoubleSignal::Spectrum);
  return sig;
}

void ComplexSignal::debugPrint() const
{
  TRACE;
  printf( "Debug samples for signal %s\n", debugName().toLatin1().data());
  CONST_LOCK_SAMPLES(Complex, thisSamples, this)
    for(int i=0;i < _nSamples;i++ ) printf( "%i\t%lg\t%lg\n", i, thisSamples[ i ].re(), thisSamples[ i ].im());
  UNLOCK_SAMPLES(this)
}

} // namespace GeopsyCore
