/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-06
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef REALSPACE_H
#define REALSPACE_H

#include <QGpCoreTools.h>
#include "Parameter.h"
#include "DinverCoreDLLExport.h"

namespace DinverCore {

class DINVERCORE_EXPORT RealSpace : public XMLClass
{
  TRANSLATIONS( "RealSpace" )
public:
  RealSpace() {}
  virtual ~RealSpace();

  virtual const QString& xml_tagName() const {return xmlRealSpaceTag;}
  static const QString xmlRealSpaceTag;

  bool operator==(const RealSpace& o) const;
  bool operator!=(const RealSpace& o) const {return !operator==(o);}

  uint checksum() const;

  void setParameterNamePrefix(const QString& p) {_parameterNamePrefix=p;}
  Parameter * addParameter(QString name, QString unit, double min, double max,
                           ParameterGrid::Scale scale, double precision);
  Parameter * addParameter(Parameter * p);
  void setVariableParameters();
  void clearParameters();

  int variableParameterCount() const {return _variableParameters.count();}
  int allParameterCount() const {return _allParameters.count();}
  double possibleCount() const;

  const Parameter * parameter(int index) const {return _allParameters[index];}
  Parameter * parameter(int index) {return _allParameters[index];}
  const Parameter * variableParameter(int index) const {return _variableParameters[index];}
  Parameter * variableParameter(int index) {return _variableParameters[index];}
  Parameter * parameter(QString name);

  inline bool isOk() const;
  inline bool isOkDebug() const;
  bool isOkDebugVerbose() const;
  void conditionDiagnostic() const;
  void humanInfo() const;

  void addCondition(AbstractCondition * c);
  bool adjustRanges();
  QString toString() const;
  QString gridToString() const;
private:
  // List of all parameters including the fixed ones
  QVector<Parameter *> _allParameters;
  // List of not fixed parameters
  QVector<Parameter *> _variableParameters;
  // Kept here only for clean delete
  QVector<AbstractCondition *> _conditions;
  QString _parameterNamePrefix;
protected:
  virtual void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
  virtual bool xml_polish(XML_POLISH_ARGS);
};

inline bool RealSpace::isOk() const
{
  foreach(Parameter * p, _allParameters) {
    if( !p->isOk()) return false;
  }
  return true;
}

inline bool RealSpace::isOkDebug() const
{
  foreach(Parameter * p, _allParameters) {
    if( !p->isOkDebug()) return false;
  }
  return true;
}

} // namespace DinverCore

#endif // REALSPACE_H
