/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-04-11
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "PdfCurve.h"
#include "Parameter.h"
#include "NewModel.h"

namespace DinverCore {

  /*!
    \class PdfPoint PdfPoint.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    \class PdfCurve Pdfcurve.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    The number of double entries is very limited and exceptional
     The curve is normally sorted, we just check.
     Curve::sort() is not selecting the correct samples in case of duplicates

     Double x entries may occur in case of ambiguities (\sa VoronoiNavigator::intersections())
     Amiguities are met when one cell touches the axis with just one vertex
     By misfortune, the one vertex cell may be selected which results with a null size step in this pdf function.
  */
  void PdfCurve::sort()
  {
    int n=count()-1;
    ASSERT(n>=0);
    for(int i=0; i<n; i++) {
      double x1=at(i).x();
      double x2=at(i+1).x();
      if(x1>=x2) {
        if(fabs(x1-x2)<=x1*1e-10) { // tolerate some inaccuracy
          remove(i);
          i--;
          n--;
        } else {
          debugPrint();
          ASSERT(false);
        }
      }
    }
    ASSERT(!constAt(count()-1).cell().isValid());
    _sorted=true;
  }

  void PdfCurve::setProbability(const ActiveModels& models, double degreesoffreedom)
  {
    for(int i=count()-1; i>=0; i--) {
      PdfPoint& p=constXAt(i);
      p.setProbability(models, degreesoffreedom);
    }
  }

  void PdfCurve::setAcceptable(const ActiveModels& models, double minimumMisfit)
  {
    for(int i=count()-2; i>=0; i--) {
      PdfPoint& p=constXAt(i);
      p.setAcceptable(models, minimumMisfit);
    }
    //debugPrint(&models);
  }

  void PdfCurve::setBorders()
  {
    int n=count()-1;
    if(n<2) {  // Either already 1 or 0, nothing to do.
      return;
    }
    int i1, i2;
    for(i1=0; i1<n; i1++) {
      if(constAt(i1).value()==1.0) {
        i1--;
        if(i1<0) {
          i1=0;
        }
        for(i2=i1+1; i2<n; i2++) {
          if(constAt(i2).value()==0.0) {
            setBorderShapeDensity(i1, i2);
            i1=i2;
            break;
          }
        }
        if(i2==n) {
          setBorderShapeDensity(i1, n-1);
          break;
        }
      }
    }
    //debugPrint();
  }

  void PdfCurve::setStrictBorders()
  {
    int n=count()-1;
    if(n<2) {  // Either already 1 or 0, nothing to do.
      return;
    }
    int i1, i2;
    for(i1=0; i1<n; i1++) {
      if(constAt(i1).value()==1.0) {
        for(i2=i1+1; i2<n; i2++) {
          if(constAt(i2).value()==0.0) {
            i2--;  // i2 is always >=1 before decrementing
            setStrictBorderShapeDensity(i1, i2);
            i1=i2;
            break;
          }
        }
        if(i2==n) {
          setStrictBorderShapeDensity(i1, n-1);
          break;
        }
      }
    }
    //debugPrint();
  }

  void PdfCurve::hits(NewModel * m)
  {
    for(int i=count()-2; i>=0; i--) {
      const PdfPoint& p=constAt(i);
      if(p.value()>0.0) {
        m->addNavigatorHit(p.cell());
      }
    }
  }

  /*!
    Linear probability over max 10% from border.
    Corrected by the width of each cell to set a higher probability for
    less dense areas.
  */
  void PdfCurve::setBorderShape(int i1, int i2)
  {
    ASSERT(i2-i1>0);
    int i=i1;
    constXAt(i).setValue(1.0);
    i++;
    constXAt(i).setValue(1.0);
    i++;
    if(i<i2) {
      double x1=constAt(i1).x();
      double x2=constAt(i2).x();
      double width=0.1*(x2-x1);
      double invWidth=1.0/width;
      double max=x1+width;
      double x0=x(i);
      while(x0<max) {
        constXAt(i).setValue((1.0-(x0-x1)*invWidth));
        x0=x(++i);
      }
      max=x2-width;
      while(x0<max) {
        constXAt(i).setValue(0.0);
        x0=x(++i);
      }
      int i21=i2-1;
      while(i<i21) {
        constXAt(i).setValue(((x(i)-max)*invWidth));
        i++;
      }
    }
    constXAt(i2-1).setValue(1.0);
    constXAt(i2).setValue(1.0);
  }

  /*!
    Linear probability over max 10% from border.
    Corrected by the width of each cell to set a higher probability for
    less dense areas.
  */
  void PdfCurve::setBorderShapeDensity(int i1, int i2)
  {
    ASSERT(i1<i2);
    int i=i1;
    constXAt(i).setValue(constAt(i+1).x()-x(i));
    i++;
    constXAt(i).setValue(constAt(i+1).x()-x(i));
    i++;
    if(i<i2) {
      double x1=x(i1);
      double x2=x(i2);
      double width=0.1*(x2-x1);
      double invWidth=1.0/width;
      double max=x1+width;
      double x0=x(i), w;
      while(x0<max) {
        w=x(i+1)-x0;
        constXAt(i).setValue((1.0-(x0-x1)*invWidth)*w);
        x0=x(++i);
      }
      max=x2-width;
      while(x0<max) {
        constXAt(i).setValue(0.0);
        x0=x(++i);
      }
      int i21=i2-1;
      while(i<i21) {
        x0=x(i);
        w=x(i+1)-x0;
        constXAt(i).setValue(((x0-max)*invWidth)*w);
        i++;
      }
    }
    constXAt(i2-1).setValue(x(i2)-x(i2-1));
    constXAt(i2).setValue(x(i2+1)-x(i2));
  }

  /*!
    Linear probability over max 10% from border.
    Corrected by the width of each cell to set a higher probability for
    less dense areas.
  */
  void PdfCurve::setStrictBorderShapeDensity(int i1, int i2)
  {
    ASSERT(i1<=i2);
    ASSERT(i2+1<count());
    constXAt(i1).setValue(constAt(i1+1).x()-x(i1));
    int i=i1;
    i++;
    if(i<i2) {
      double x1=x(i1);
      double x2=x(i2);
      double width=0.1*(x2-x1);
      double invWidth=1.0/width;
      double max=x1+width;
      double x0=x(i), w;
      while(x0<max) {
        w=x(i+1)-x0;
        constXAt(i).setValue((1.0-(x0-x1)*invWidth)*w);
        x0=x(++i);
      }
      max=x2-width;
      while(x0<max) {
        constXAt(i).setValue(0.0);
        x0=x(++i);
      }
      while(i<i2) {
        x0=x(i);
        w=x(i+1)-x0;
        constXAt(i).setValue(((x0-max)*invWidth)*w);
        i++;
      }
    }
    constXAt(i2).setValue(x(i2+1)-x(i2));
  }

  void PdfCurve::setBorderShapeCosine(int i1, int i2)
  {
    /*for(int i=i1; i<=i2; i++) {
      double x=operator[](i).x();
      operator[](i).setValue(1.0+cos((operator[](i).x()-x1)*period));
    }*/
  }

  void PdfCurve::debugPrint(const ActiveModels * models) const
  {
    TRACE;
    int n=count()-1;
    if(models) {
      for(int i=0; i<n; i++) {
        const PdfPoint& p=at(i);
        printf("%5i %20.16lf %10lf %10lf %10i\n",
               i,
               p.x(),
               p.value(),
               models->misfit(p.cell(), TargetIndex::first),
               p.cell().value());
      }
    } else {
      for(int i=0; i<n; i++) {
        const PdfPoint& p=at(i);
        printf("%5i %20.16lf %10lf %10i\n",
               i,
               p.x(),
               p.value(),
               p.cell().value());
      }
    }
    printf("%5i %10lf\n", n, x(count()-1));
  }
  double * PdfCurve::cumulativeDistribution() const
  {
    int n=count();
    double * cum=new double[n];
    cum[0]=0.0;
    const PdfPoint * p1, *p2;
    p2=&constAt(0);
    for(int i=1; i<n; i++) {
      p1=p2;
      p2=&at(i);
      cum[i]=cum[i-1]+(p2->x()-p1->x())*p1->value();
    }
    return cum;
  }

  void PdfCurve::printCumulativeDistribution(const Parameter& p)
  {
    TRACE;
    double * cum=cumulativeDistribution();
    int n=count();
    for(int i=1; i<n; i++) {
      printf("%i %lf %lg\n", i, p.realValue(qRound(at(i).x())), cum[i]/cum[n-1]);
    }
    delete [] cum;
  }

  /*!
    For debug purpose only.
    Compares two pdf curves.
  */
  bool PdfCurve::compare(const PdfCurve& o, int& index) const
  {
    if(count()!=o.count()) {
      debugPrint();
      o.debugPrint();
      return false;
    }
    for(int i=count()-1; i>=0; i--) {
      if(at(i).x()!=o.at(i).x() ||
         at(i).cell().value()!=o.at(i).cell().value()) {
        printf("Difference found at index %i\n", i);
        debugPrint();
        o.debugPrint();
        index=i;
        return false;
      }
    }
    return true;
  }

  /*!
    Generate a random value that follow the probability dentsity function given by \a pdf.
    \a pdf is not necessarily normalized. It is defined by constant steps:
    \code
    pdf[x]       pdf[y]
    25           23          23 if 25 &lt; x &lt; 30
    30           35          35 if 30 &lt; x &lt; 35
    35           45          45 if 35 &lt; x &lt; 37
    37           50          50 if 37 &lt; x &lt; 41
    41           20          20 if 41 &lt; x &lt; 50
    50           10          10 if 50 &lt; x &lt; 70
    70           10          last value not used, only the x is used for max of last class
    \endcode

    \a randomValue must be between 0 and 1.
  */
  int PdfCurve::generate(double randomValue, ActiveIndex& cellIndex)
  {
    TRACE;
    int n=count();
    double * cum=cumulativeDistribution();
    /*for(int i=0; i<n; i++) {
      printf("%5i %30.16lf %10lg %5i\n", i, at(i).x(), cum[i], at(i).cell().value());
    }*/
    double val=randomValue*cum[n-1];
    n--;
    if(cum[n]==0.0) {
      cellIndex=ActiveIndex::null;
      return 0.0;
    }
    int index;
    if(n<12) {
      for(index=1; index<n; index++) {
        if(val<cum[index]) break;
      }
    } else {
      int n2=1;
      while(n2<n) n2=n2 << 1; // multiply by 2
      index=n2;
      int step2=index >> 1;
      while(step2>0) {
        if(index>n) index-=step2;
        else if(val<=cum[index] ) {
          if(val>cum[index-1] ) break;
          index-=step2;
        } else
          index+=step2;
        step2=step2 >> 1;
      }
    }
    // Rounds here to set the correct cell index
    // Rounding may shift cell index
    index--;
    val=round(at(index).x()+(val-cum[index])
        /(cum[index+1]-cum[index] )
         *(at(index+1).x()-at(index).x()));
    while(index<n-1 && val>at(index+1).x()) {
      index++;
    }
    while(index>0 && val<=at(index).x()) {
      index--;
    }
    cellIndex=at(index).cell();
    //printf("val %30.16lf cell %5i\n", val, cellIndex.value());
    return val;
  }

} // namespace DinverCore
