/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-02-06
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef ARRAYPARAMETERS_H
#define ARRAYPARAMETERS_H

#include <GeopsyCore.h>

#include "ArrayCoreDLLExport.h"
#include "ArraySelection.h"
#include "BlockAveragingParameters.h"

namespace ArrayCore {

  class ARRAYCORE_EXPORT ArrayParameters: public AbstractParameters
  {
  public:
    ArrayParameters();
    ArrayParameters(const ArrayParameters& o);

    TimeRangeParameters& timeLimits() {return _timeLimits;}
    const TimeRangeParameters& timeLimits() const {return _timeLimits;}

    WindowingParameters& windowing() {return _windowing;}
    const WindowingParameters& windowing() const {return _windowing;}

    SamplingParameters& frequencySampling() {return _frequencySampling;}
    const SamplingParameters& frequencySampling() const {return _frequencySampling;}

    BlockAveragingParameters& blockAveraging() {return _blockAveraging;}
    const BlockAveragingParameters& blockAveraging() const {return _blockAveraging;}

    WindowFunctionParameters& taper() {return _taper;}
    const WindowFunctionParameters& taper() const {return _taper;}

    double frequencyBandwidth() const {return _frequencyBandwidth;}
    void setfrequencyBandwidth(double f) {_frequencyBandwidth=f;}

    double oversamplingFactor() const {return _oversamplingFactor;}
    void setOversamplingFactor(double f) {_oversamplingFactor=f;}

    const QString& outputBaseName() const {return _outputBaseName;}
    void setOutputBaseName(const QString& n) {_outputBaseName=n;}
    virtual QString outputName(const QString& groupName) const=0;

    double selectDurationFactor() const {return _selectDurationFactor;}
    void setSelectDurationFactor(double f) {_selectDurationFactor=f;}

    const Point2D& selectArrayCenter() const {return _selectArrayCenter;}
    void setSelectArrayCenter(const Point2D& c) {_selectArrayCenter=c;}

    double selectArrayRadius() const {return _selectArrayRadius;}
    void setSelectArrayRadius(double f) {_selectArrayRadius=f;}

    double magneticDeclination() const {return _magneticDeclination;}
    void setMagneticDeclination(double d) {_magneticDeclination=d;}

    inline void acceptAllFrequencies();
    void setMinimumFrequency(double frequency) {_minimumFrequency=frequency*(1.0-1e-10);}
    void setMaximumFrequency(double frequency) {_maximumFrequency=frequency*(1.0+1e-10);}
    inline bool acceptFrequency(double frequency) const;

    virtual void selectStations(ArraySelection& array) const;

    virtual QString toString(PARAMETERS_TOSTRING_ARGS_DECL) const;
    virtual void setVersion(PARAMETERS_SETVERSION_ARGS);
    virtual void collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS);
    virtual int totalKeywordCount(PARAMETERS_TOTALKEYWORDCOUNT_ARGS) const;
  protected:
    virtual int keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const;
    virtual bool setValue(PARAMETERS_SETVALUE_ARGS);
  private:
    TimeRangeParameters _timeLimits;
    WindowingParameters _windowing;
    SamplingParameters _frequencySampling;
    BlockAveragingParameters _blockAveraging;
    WindowFunctionParameters _taper;
    double _selectDurationFactor;
    Point2D _selectArrayCenter;
    double _selectArrayRadius;
    double _magneticDeclination;
    double _minimumFrequency;
    double _maximumFrequency;
    double _frequencyBandwidth;
    double _oversamplingFactor;
    QString _outputBaseName;
  };

  inline void ArrayParameters::acceptAllFrequencies()
  {
    TRACE;
    _minimumFrequency=0.0;
    _maximumFrequency=std::numeric_limits<double>::infinity();
  }

  inline bool ArrayParameters::acceptFrequency(double frequency) const
  {
    TRACE;
    return frequency>=_minimumFrequency &&
           frequency<=_maximumFrequency;
  }

} // namespace ArrayCore

#endif // ARRAYPARAMETERS_H

