/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2012-03-27
**  Copyright: 2012-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ActiveFKParameters.h"

namespace ArrayCore {

  /*!
    \class ActiveFKParameters ActiveFKParameters.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  ActiveFKParameters::ActiveFKParameters()
    : FKParameters()
  {
    TRACE;
    timeLimits().setStartType(TimeRangeParameters::Signal);
    timeLimits().setEndType(TimeRangeParameters::Signal);

    windowing().setLength(2.0);
    windowing().setSeismicEventTrigger(true);
    windowing().setSeismicEventDelay(0.0);

    setMinimumDistance(10.0);
    setMaximumDistance(200.0);

    setOversamplingFactor(4.0);
    setProcessType(Conventional);

    frequencySampling().setScaleType(SamplingParameters::Log);
    frequencySampling().setRange(10.0, 50.0);
    frequencySampling().setStep(1.025);

    blockAveraging().setCount(1);

    _ySampling.setScaleType(SamplingParameters::Log);
    _ySampling.setStep(1.01);
    _ySampling.setRange(100.0, 1000.0);

    setMaximumSlowness(1.0/100.0);

    _beamPowerNormalization=Spectrum;
  }

  ActiveFKParameters::ActiveFKParameters(const ActiveFKParameters& o)
    : FKParameters(o),
      _ySampling(o._ySampling)
  {
    TRACE;
    _beamPowerNormalization=o._beamPowerNormalization;
  }

  void ActiveFKParameters::setBeamPowerNormalization(const QString& b)
  {
    TRACE;
    if(b.count()>0) {
      QString bl=b.toLower();
      switch(bl[0].unicode()) {
      case 'r':
        if(bl=="raw") {
          _beamPowerNormalization=Raw;
        }
        break;
      case 's':
        if(bl=="spectrum") {
          _beamPowerNormalization=Spectrum;
        }
        break;
      case 'm':
        if(bl=="maximum") {
          _beamPowerNormalization=Maximum;
        }
        break;
      default:
        break;
      }
    }
  }

  QString ActiveFKParameters::beamPowerNormalizationString() const
  {
    TRACE;
    switch(_beamPowerNormalization) {
    case Raw:
      break;
    case Spectrum:
      return "Spectrum";
    case Maximum:
      return "Maximum";
    }
    return "Raw";
  }

  void ActiveFKParameters::setVersion(PARAMETERS_SETVERSION_ARGS)
  {
    TRACE;
    _ySampling.setVersion(version);
    FKParameters::setVersion(version);
  }

  int ActiveFKParameters::totalKeywordCount(PARAMETERS_TOTALKEYWORDCOUNT_ARGS) const
  {
    return FKParameters::totalKeywordCount()+
        _ySampling.totalKeywordCount();
  }

  int ActiveFKParameters::keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const
  {
    return 1+FKParameters::keywordCount();
  }

  void ActiveFKParameters::collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS)
  {
    TRACE;
    int baseIndex=FKParameters::keywordCount();
    FKParameters::collectKeywords(keywords, prefix, suffix);
    keywords.add(prefix+"BEAM POWER NORMALIZATION"+suffix, this, baseIndex);
    _ySampling.collectKeywords(keywords, prefix+"Y ", suffix);
  }

  QString ActiveFKParameters::toString(PARAMETERS_TOSTRING_ARGS_IMPL) const
  {
    TRACE;
    QString log;
    log+=FKParameters::toString(prefix, suffix);
    log+=_ySampling.toString(prefix+"Y ", suffix);
    log+=prefix+"BEAM POWER NORMALIZATION"+suffix+"="+beamPowerNormalizationString()+"\n";
    return log;
  }

  bool ActiveFKParameters::setValue(PARAMETERS_SETVALUE_ARGS)
  {
    TRACE;
    switch(index-FKParameters::keywordCount()) {
    case 0:
      setBeamPowerNormalization(value);
      return true;
    default:
      break;
    }
    return FKParameters::setValue(index, value, unit, keywords);
  }

} // namespace ArrayCore
