/***************************************************************************
**
**  This file is part of waran.
**
**  waran is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  waran is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2012-02-04
**  Copyright: 2012-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "AcquisitionWidget.h"
#include "NetworkParameterEditor.h"
#include "SeismicParameterEditor.h"
#include "StationItem.h"
#include "Station.h"
#include "DownloadGpsRaw.h"
#include "StartGps.h"

/*!
  \class AcquisitionWidget AcquisitionWidget.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
AcquisitionWidget::AcquisitionWidget(QWidget * parent)
  : QWidget(parent)
{
  TRACE;
  setupUi(this);

  StationItem * stationModel=new StationItem(this);
  connect(stationModel, SIGNAL(dataChanged(QModelIndex,QModelIndex)), &_resizeColumns, SLOT(newEvent()));
  unitParameters->setModel(stationModel);
  unitParameters->setSelectionMode(QAbstractItemView::ExtendedSelection);
  unitParameters->setSelectionBehavior(QAbstractItemView::SelectRows);
  unitParameters->setEditTriggers(QAbstractItemView::NoEditTriggers);
  _networkParameters.restore();
  _resizeColumns.setInterval(10000);
  connect(&_resizeColumns, SIGNAL(run()), this, SLOT(adjustColumnWidth()));
}

/*!
  Description of destructor still missing
*/
AcquisitionWidget::~AcquisitionWidget()
{
  TRACE;
  _networkParameters.save();
}

void AcquisitionWidget::setStations(QList<Station *> * list)
{
  TRACE;
  _stations=list;
  static_cast<StationItem *>(unitParameters->model())->setStations(_stations);
}

void AcquisitionWidget::adjustColumnWidth()
{
  TRACE;
  unitParameters->resizeColumnsToContents();
}

bool AcquisitionWidget::checkSelection(const QString& title)
{
  TRACE;
  QItemSelectionModel * model=unitParameters->selectionModel();
  for(QList<Station *>::iterator it=_stations->begin(); it!=_stations->end(); it++) {
    if(model->isRowSelected((*it)->index(), QModelIndex())) {
      return true;
    }
  }
  if(Message::question(MSG_ID, title, tr("No station is currently selected, do you want to select all of them?"),
                       Message::yes(), Message::no())==Message::Answer1) {
    return false;
  } else {
    unitParameters->selectAll();
    return true;
  }
}

void AcquisitionWidget::on_seismicStartButton_clicked()
{
  TRACE;
  if(!checkSelection(tr("Starting seismic acquisition"))) return;
  QItemSelectionModel * model=unitParameters->selectionModel();
  for(QList<Station *>::iterator it=_stations->begin(); it!=_stations->end(); it++) {
    if(model->isRowSelected((*it)->index(), QModelIndex()) && (*it)->type()==Station::AcquisitionUnit) {
      (*it)->startSeismic();
    }
  }
}

void AcquisitionWidget::on_seismicStopButton_clicked()
{
  TRACE;
  if(!checkSelection(tr("Stopping seismic acquisition"))) return;
  QItemSelectionModel * model=unitParameters->selectionModel();
  for(QList<Station *>::iterator it=_stations->begin(); it!=_stations->end(); it++) {
    if(model->isRowSelected((*it)->index(), QModelIndex()) && (*it)->type()==Station::AcquisitionUnit) {
      (*it)->stopSeismic();
    }
  }
}

void AcquisitionWidget::on_gpsStartButton_clicked()
{
  TRACE;
  QString title=tr("Starting gps acquisition");
  if(!checkSelection(title)) return;
  StartGps * d=new StartGps(this);
  Settings::getWidget(d);
  if(d->exec()==Dialog::Accepted) {
    Settings::setWidget(d);
    QItemSelectionModel * model=unitParameters->selectionModel();
    for(QList<Station *>::iterator it=_stations->begin(); it!=_stations->end(); it++) {
      if(model->isRowSelected((*it)->index(), QModelIndex()) && (*it)->type()==Station::AcquisitionUnit) {
        (*it)->startGps(d->duration());
      }
    }
  }
  delete d;
}

void AcquisitionWidget::on_gpsStopButton_clicked()
{
  TRACE;
  QString title=tr("Stopping gps acquisition");
  if(!checkSelection(title)) return;
  QItemSelectionModel * model=unitParameters->selectionModel();
  for(QList<Station *>::iterator it=_stations->begin(); it!=_stations->end(); it++) {
    if(model->isRowSelected((*it)->index(), QModelIndex()) && (*it)->type()==Station::AcquisitionUnit) {
      (*it)->stopGps();
    }
  }
}

void AcquisitionWidget::on_gpsNavigationButton_clicked()
{
  TRACE;
  if(!checkSelection(tr("Requesting navigation data"))) return;
  QItemSelectionModel * model=unitParameters->selectionModel();
  for(QList<Station *>::iterator it=_stations->begin(); it!=_stations->end(); it++) {
    if(model->isRowSelected((*it)->index(), QModelIndex()) && (*it)->type()==Station::AcquisitionUnit) {
      (*it)->navigationGps();
    }
  }
}

void AcquisitionWidget::on_gpsDownloadButton_clicked()
{
  TRACE;
  if(!checkSelection(tr("Download raw GPS data"))) return;
  // Creates working directory
  QDir wd("/tmp/waran");
  if(!wd.exists()) {
    wd.setPath("/tmp");
    wd.mkdir("waran");
  }

  DownloadGpsRaw * d=new DownloadGpsRaw(this);
  QItemSelectionModel * model=unitParameters->selectionModel();
  for(QList<Station *>::iterator it=_stations->begin(); it!=_stations->end(); it++) {
    if(model->isRowSelected((*it)->index(), QModelIndex()) && (*it)->type()==Station::AcquisitionUnit) {
      (*it)->downloadGps();
      d->addStation(*it);
    }
  }
  if(d->exec()==QDialog::Accepted) {
    d->toRinex();
  } else {
    d->terminate();
  }
  delete d;
}

void AcquisitionWidget::on_rebootButton_clicked()
{
  TRACE;
  if(!checkSelection(tr("Reboot"))) return;
  QItemSelectionModel * model=unitParameters->selectionModel();
  for(QList<Station *>::iterator it=_stations->begin(); it!=_stations->end(); it++) {
    if(model->isRowSelected((*it)->index(), QModelIndex())) {
      (*it)->reboot();
    }
  }
}

void AcquisitionWidget::on_powerOffButton_clicked()
{
  TRACE;
  if(!checkSelection(tr("Power off"))) return;
  QItemSelectionModel * model=unitParameters->selectionModel();
  for(QList<Station *>::iterator it=_stations->begin(); it!=_stations->end(); it++) {
    if(model->isRowSelected((*it)->index(), QModelIndex())) {
      (*it)->powerOff();
    }
  }
}

void AcquisitionWidget::on_clearUsbButton_clicked()
{
  TRACE;
  if(Message::question(MSG_ID, tr("Clear USB"),
                       tr("Do you really want to clear all files older than 5 days?"),
                       Message::yes(), Message::no())==Message::Answer1) {
    return;
  }
  if(!checkSelection(tr("Clear USB"))) return;
  QItemSelectionModel * model=unitParameters->selectionModel();
  for(QList<Station *>::iterator it=_stations->begin(); it!=_stations->end(); it++) {
    if(model->isRowSelected((*it)->index(), QModelIndex())) {
      (*it)->clearUsb();
    }
  }
}

void AcquisitionWidget::on_networkParameters_clicked()
{
  TRACE;
  NetworkParameterEditor * d=new NetworkParameterEditor(this);
  Settings::getRect(d, "NetworkParameterEditor");
  d->setValues(_networkParameters);
  if(d->exec()==QDialog::Accepted) {
    Settings::setRect(d, "NetworkParameterEditor");
    d->getValues(_networkParameters);
    emit networkParametersChanged();
  }
  delete d;
}

void AcquisitionWidget::on_seismicParameters_clicked()
{
  TRACE;
  MessageContext mc;
  if(!checkSelection(tr("Setting seismic parameters"))) return;
  SeismicParameterEditor * d=new SeismicParameterEditor(this);
  Settings::getWidget(d);
  d->updateFields();
  if(d->exec()==QDialog::Accepted) {
    Settings::setWidget(d);
    SeismicParameters param;
    d->getValues(param);
    // Apply to selected stations
    QItemSelectionModel * model=unitParameters->selectionModel();
    for(QList<Station *>::iterator it=_stations->begin(); it!=_stations->end(); it++) {
      if(model->isRowSelected((*it)->index(), QModelIndex()) && (*it)->type()==Station::AcquisitionUnit) {
        (*it)->setSeismicParameters(param);
      }
    }
  }
  delete d;
}
