/***************************************************************************
**
**  This file is part of gpviewmax.
**
**  gpviewmax is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpviewmax is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-07-16
**  Copyright: 2018-2019

**
***************************************************************************/

#include <SciFigs.h>

#include "gpviewmaxVersion.h"
#include "gpviewmaxInstallPath.h"
#include "HistogramsWidget.h"
#include "Reader.h"

PACKAGE_INFO("gpviewmax", GPVIEWMAX)

ApplicationHelp * help();

HistogramsWidget * createWidgets(Reader& reader)
{
  HistogramsWidget * w=new HistogramsWidget;
  w->setNormalize(reader.isNormalized());
  w->setSamples(*reader.samples(), reader);
  w->setHistogramComments(reader.fileNames().join(";"));
  if(reader.fileNames().count()==1) {
    QFileInfo fi(reader.fileNames().first());
    w->setWindowTitle(fi.fileName()+" - gpviewmax");
  }
  if(!w->splitModes(*reader.samples(), reader)) {
    delete w;
    return nullptr;
  }
  return w;
}

int main(int argc, char ** argv)
{
  Application a(argc, argv, help);

  // Options
  ExportOptions exportOptions;
  Reader reader;
  reader.setSamples(new Samples);
  if(!reader.setOptions(argc, argv)) {
    return 2;
  }
  if(!exportOptions.read(argc, argv)) {
    return 2;
  }
  if(!CoreApplication::checkRemainingArgs(argc, argv)) {
    return 2;
  }
  if(!reader.read(argc, argv, reader.ignoreStdin())) {
    return 2;
  }
  if(reader.isEmpty()) {
    if(!reader.read(Message::getOpenFileNames(tr("Open data file"),
                                              tr("All files (*);;"
                                                 "Max files (*.max)")))) {
      return 2;
    }
  }
  // User basic filtering
  Samples * samples;
  if(!reader.nullEllipticities()) {
    samples=reader.samples()->filterNullEllipticities();
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  if(reader.curveFileNames().isEmpty()) {
    samples=reader.samples()->filterNoiseSmart(reader.noiseDeviation(),
                                               reader.slownessDeviation());
    if(samples) {
      delete reader.samples();
      reader.setSamples(samples);
    }
  }
  samples=reader.samples()->filterNoise(reader.noiseThreshold());
  if(samples) {
    delete reader.samples();
    reader.setSamples(samples);
  }
  samples=reader.samples()->filterRelativePower(reader.relativePowerThreshold());
  if(samples) {
    delete reader.samples();
    reader.setSamples(samples);
  }

  if(!reader.setDefaultLimits()) {
    return 2;
  }

  int appReturn;
  switch(reader.action()) {
  case HistogramReader::Gui: {
      SciFigsGlobal s;
      HistogramsWidget * w=createWidgets(reader);
      if(!w) {
        return 2;
      }
      w->exportPlot(exportOptions);
      w->show();
      appReturn=a.exec();
      delete w;
    }
    break;
  case HistogramReader::Max: {
      reader.samples()->save(reader.exportMaxFileName(), reader.headerLine(), reader.polarization());
      appReturn=0;
    }
    break;
  case HistogramReader::Plot: {
      SciFigsGlobal s;
      HistogramsWidget * w=createWidgets(reader);
      if(!w) {
        return 2;
      }
      w->exportPlot(exportOptions);
      appReturn=0;
      delete w;
    }
    break;
  default:
    appReturn=2;
    break;
  }
  return appReturn;
}

ApplicationHelp * help()
{
  TRACE;
  ApplicationHelp * h=new ApplicationHelp;
  h->setOptionSummary( "[OPTIONS] <FILE>...");
  h->setComments("Plot all values of a .max file generated with three-component FK. "
                 "Plot of individual modes.");
  h->addGroup("polarisation", "pol");
  h->addOption("-R", "Plot only Rayleigh, null ellipticities are filtered out unless option '-null-ell' is specified.");
  h->addOption("-L", "Plot only Love.");
  h->addGroup("filter", "filter");
  h->addOption("-rel-power <R>", "Accept only samples with power > R*maximum power (over a time window and for a frequency).");
  h->addOption("-max-noise <R>", "Accept only samples with relative incoherent noise < R.");
  h->addOption("-curve-rel-range <R>", "Keep R*mean around curves to split modes (default=0.2).");
  h->addOption("-null-ell", "Keep null ellipticities (default=filter them).");
  h->addOption("-low-noise <N> <S>", "Keep only low noise samples in a slowness neighborhood. "
                                     "For a given frequency, each sample S0, N0 is considered individually. "
                                     "If there is at least one sample Si, Ni in the slowness neighborhood "
                                     "defined by [(1-S)*S0, (1+S)*S0] for which N0 is larger than (1+N)*Ni, "
                                     "the current sample is filtered out.");
  h->addGroup("Output", "output");
  h->addOption("-plot", "Export plot (see '-h plot' for options).");
  h->addOption("-max <FILE>", "Output filter max values and exit.");
  HistogramReader::helpCurves(h);
  HistogramReader::helpX(h);
  Reader::helpY(h);
  h->addGroup("Plot", "plot", 2);
  ExportOptions::help(h);
  return h;
}
