/***************************************************************************
**
**  This file is part of gpfksimulator.
**
**  gpfksimulator is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpfksimulator is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2011-01-06
**  Copyright: 2011-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "SourceItemModel.h"

/*!
  \class SourceItemModel SourceItemModel.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
SourceItemModel::SourceItemModel(QList<SourceParameters> * sources, QObject * parent)
  : QAbstractTableModel(parent)
{
  TRACE;
  _sources=sources;
  _waveModel=SourceParameters::PlaneWaves;
}

/*!
  Description of destructor still missing
*/
SourceItemModel::~SourceItemModel()
{
  TRACE;
}

void SourceItemModel::setWaveModel(SourceParameters::WaveModel wm)
{
  TRACE;
  beginResetModel();
  _waveModel=wm;
  endResetModel();
}

int SourceItemModel::rowCount(const QModelIndex &) const
{
  TRACE;
  return _sources->count();
}

/*!
*/
int SourceItemModel::columnCount(const QModelIndex &) const
{
  TRACE;
  switch(_waveModel) {
  case SourceParameters::PlaneWaves:
    break;
  case SourceParameters::CircularWaves:
    return 11;
  }
  return 8;
}

QVariant SourceItemModel::data(const QModelIndex &index, int role) const
{
  TRACE;
  if(!index.isValid()) return QVariant();
  const SourceParameters& p=_sources->at(index.row());
  switch (role) {
  case Qt::DisplayRole:
    switch(_waveModel) {
    case SourceParameters::PlaneWaves:
      switch(index.column()) {
      case 0: return p.name();
      case 1: return Mode::userPolarization(p.polarization());
      case 2: return p.azimuth(_waveModel);
      case 3: return p.amplitude();
      case 4: return Angle::radiansToDegrees(p.phase());
      case 5: return p.velocity();
      case 6: return p.ellipticity();
      case 7: {
          QColor col;
          guiColor(p.color(), col);
          return col;
        }
      default: return QVariant();
      }
    case SourceParameters::CircularWaves:
      switch(index.column()) {
      case 0: return p.name();
      case 1: return Mode::userPolarization(p.polarization());
      case 2: return p.azimuth(_waveModel);
      case 3: return p.distance();
      case 4: return p.sourcePoint().x();
      case 5: return p.sourcePoint().y();
      case 6: return p.amplitude();
      case 7: return Angle::radiansToDegrees(p.phase());
      case 8: return p.velocity();
      case 9: return p.ellipticity();
      case 10: {
          QColor col;
          guiColor(p.color(), col);
          return col;
        }
      default: return QVariant();
      }
    }
  default:
    return QVariant();
  }
}

bool SourceItemModel::setData (const QModelIndex & index, const QVariant & value, int role)
{
  TRACE;
  SourceParameters& p=(*_sources)[index.row()];
  switch(role) {
  case Qt::EditRole:
    switch(_waveModel) {
    case SourceParameters::PlaneWaves:
      switch(index.column()) {
      case 0: p.setName(value.toString()); break;
      case 1: p.setPolarization(Mode::userPolarization(value.toString())); break;
      case 2: p.setAzimuth(_waveModel, value.toDouble()); break;
      case 3: p.setAmplitude(value.toDouble()); break;
      case 4: p.setPhase(Angle::degreesToRadians(value.toDouble())); break;
      case 5: p.setVelocity(value.toDouble()); break;
      case 6: p.setEllipticity(value.toDouble()); break;
      case 7: {
          Color col;
          coreColor(value.value<QColor>(), col);
          p.setColor(col);
          break;
        }
      default: break;
      }
      emit dataChanged(index, index);
      break;
    case SourceParameters::CircularWaves:
      switch(index.column()) {
      case 0: p.setName(value.toString()); break;
      case 1: p.setPolarization(Mode::userPolarization(value.toString())); break;
      case 2: p.setAzimuth(_waveModel, value.toDouble()); break;
      case 3: p.setDistance(value.toDouble()); break;
      case 4: p.setSourceX(value.toDouble()); break;
      case 5: p.setSourceY(value.toDouble()); break;
      case 6: p.setAmplitude(value.toDouble()); break;
      case 7: p.setPhase(Angle::degreesToRadians(value.toDouble())); break;
      case 8: p.setVelocity(value.toDouble()); break;
      case 9: p.setEllipticity(value.toDouble()); break;
      case 10:  {
          Color col;
          coreColor(value.value<QColor>(), col);
          p.setColor(col);
          break;
        }
      default: break;
      }
      emit dataChanged(createIndex(index.row(), 0), createIndex(index.row(), 9));
      break;
    }
    return true;
  default:
    return false;
  }
}
QVariant SourceItemModel::headerData(int section, Qt::Orientation orientation, int role) const
{
  TRACE;
  if(role!=Qt::DisplayRole) return QVariant();
  if(orientation==Qt::Horizontal) {
    switch(_waveModel) {
    case SourceParameters::PlaneWaves:
      switch (section) {
      case 0: return tr("Name");
      case 1: return tr("Polarization");
      case 2: return tr("Azimuth");
      case 3: return tr("Amplitude");
      case 4: return tr("Phase");
      case 5: return tr("Velocity");
      case 6: return tr("Ellipticity");
      case 7: return tr("Color");
      default: break;
      }
      break;
    case SourceParameters::CircularWaves:
      switch (section) {
      case 0: return tr("Name");
      case 1: return tr("Polarization");
      case 2: return tr("Azimuth");
      case 3: return tr("Distance");
      case 4: return tr("X");
      case 5: return tr("Y");
      case 6: return tr("Amplitude");
      case 7: return tr("Phase");
      case 8: return tr("Velocity");
      case 9: return tr("Ellipticity");
      case 10: return tr("Color");
      default: break;
      }
      break;
    }
  } else {
    return section+1;
  }
  return QVariant();
}

int SourceItemModel::colorColumn() const
{
  switch(_waveModel) {
  case SourceParameters::PlaneWaves:
    break;
  case SourceParameters::CircularWaves:
    return 10;
  }
  return 7;
}

Qt::ItemFlags SourceItemModel::flags(const QModelIndex & index) const
{
  if(index.column()==colorColumn()) {
    return Qt::ItemIsSelectable | Qt::ItemIsEnabled;
  } else {
    return Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled;
  }
}

void SourceItemModel::addSource()
{
  TRACE;
  beginInsertRows(QModelIndex(), _sources->count(), _sources->count());
  if(_sources->isEmpty()) {
    SourceParameters s;
    s.setName(tr("source_%1").arg(_sources->count()+1));
    _sources->append(s);
  } else {
    _sources->append(_sources->last());
    _sources->last().setName(tr("source_%1").arg(_sources->count()));
  }
  endInsertRows();
}

void SourceItemModel::setSources(QList<SourceParameters>& sources)
{
  TRACE;
  beginResetModel();
  *_sources=sources;
  endResetModel();
}

void SourceItemModel::removeSource(int index)
{
  TRACE;
  beginRemoveRows(QModelIndex(), index, index);
  _sources->removeAt(index);
  endRemoveRows();
}
