/***************************************************************************
**
**  This file is part of geopsy.
**
**  geopsy is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  geopsy is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-11-03
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <GeopsyGui.h>
#include <QGpGuiTools.h>
#include <SciFigs.h>

#include "MagnifierSignal.h"
#include "SignalDisplay.h"

MagnifierSignal::MagnifierSignal(QWidget* parent) :
  GraphicWindow(parent)
{
  TRACE;
  resize(200, 80);
  setMinimumSize(QSize(50,50));
  setWindowFlags(windowFlags() | Qt::WindowStaysOnTopHint);
  _signalLayer->setNormalize(SignalLayer::NormalizeVisibleOne);
  GraphContent * gc=_signalLayer->graphContent();
  AxisWindow * w=gc->graph();
  QList<QAction *> actionList=gc->actions();
  gc->removeAction(gc->action("Zoom"));
  gc->removeAction(gc->action("ZoomIn"));
  gc->removeAction(gc->action("ZoomOut"));
  gc->removeAction(gc->action("Unzoom"));
  gc->removeAction(gc->action("Edit"));
  gc->removeAction(gc->action("PickSeparator"));
  gc->removeAction(gc->action("Pick"));
  Settings::getSize(this,"MagnifierSignal");
  _cursorLayer=new ParallelBands(gc->graph());
  _cursorLayer->addBand(0,0, Brush(Qt::black, Qt::SolidPattern));
  w->xAxis()->setNumberPrecision(4);
  _context=0;
  // Init first range, user then can fix it to whatever value
  w->xAxis()->setRange(0.0, 0.1);
}

void MagnifierSignal::setSignal(SubSignalPool * parentSubPool, Signal * sig)
{
  TRACE;
  subPool().removeAll();
  if(_context <= 0) {
    subPool().addSignal(sig);
  } else {
    int iSig=parentSubPool->indexOf(sig);
    int iSigMin=iSig-_context;
    if(iSigMin<0) iSigMin =0;
    int iSigMax=iSig+_context;
    if(iSigMax>=parentSubPool->count()) iSigMax=parentSubPool->count()-1;
    for(int i=iSigMin; i<=iSigMax; i++) {
      subPool().addSignal(parentSubPool->at(i));
    }
    SignalLayer * sl=signalLayer();
    sl->clearSignalColors();
    sl->setSignalColor(sig,Qt::red);
  }
  // Save & restore current range
  AxisWindow * w=_cursorLayer->graph();
  double dt=w->xAxis()->maximum()-w->xAxis()->minimum();
  subPoolUpdate();
  w->xAxis()->setRange(0.0, dt);
}

void MagnifierSignal::setTime(double t)
{
  TRACE;
  AxisWindow * w=_cursorLayer->graph();
  double dt=0.5*(w->xAxis()->maximum()-w->xAxis()->minimum());
  w->xAxis()->setRange(t-dt,t+dt);
  ParallelBand& b=_cursorLayer->band(0);
  b.setMinimum(t);
  b.setMaximum(t);
  w->xAxis()->update();
  w->deepUpdate();
}

void MagnifierSignal::resizeEvent (QResizeEvent * )
{
  TRACE;
  _cursorLayer->graph()->setGeometry(0,0,width()-5,height());
  Settings::setSize(this,"MagnifierSignal");
}

void MagnifierSignal::setWindowTitle(QString title)
{
  TRACE;
  GraphicWindow::setWindowTitle("Magnifier - "+title);
}

void MagnifierSignal::closeEvent(QCloseEvent * e)
{
  TRACE;
  e->ignore();
}
