/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2005-07-04
**  Copyright: 2005-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "LegendProperties.h"
#include "XMLSciFigs.h"
#include "Legend.h"

namespace SciFigs {

LegendProperties::LegendProperties(QWidget * parent)
    : PropertyWidget(parent)
{
  TRACE;
  setupUi(this);
  legendTable->setLegend(&_legend);
  oneColor->setChecked(true);
  paletteEditor->setPalette(&_legend);
  connect(paletteEditor, SIGNAL(endPaletteChange()), this, SLOT(endPaletteChange()));

  addProperty(Legend, legendTable);
}

void LegendProperties::endPaletteChange()
{
  TRACE;
  if(oneColor->isChecked()) {
    _legend.penColorToSymbolPen();
    _legend.penColorToSymbolBrush();
    _legend.penColorToText();
  }
  legendTable->setLegend(&_legend);
}

PropertyValue::WidgetType LegendProperties::determineCustomWidgetType(int pid, QWidget *, QWidget * )
{
  if(pid==Legend) {
    return PropertyValue::Custom0;
  } else {
    return PropertyValue::Unknown;
  }
}

QVariant LegendProperties::customWidgetValue(PropertyValue & p)
{
  TRACE;
  if(p.widgetType()==PropertyValue::Custom0) {
    return QVariant::fromValue(_legend);
  } else {
    return QVariant();
  }
}

bool LegendProperties::connectCustomWidget(PropertyValue &p)
{
  TRACE;
  if(p.widgetType()==PropertyValue::Custom0) {
    QObject::connect(legendTable, SIGNAL(changed()), &p, SLOT(touched()) );
    return true;
  } else {
    return false;
  }
}

bool LegendProperties::setCustomWidget(PropertyValue & p)
{
  TRACE;
  if(p.widgetType()==PropertyValue::Custom0) {
    SciFigs::Legend legend=p.value().value<SciFigs::Legend>();
    if(!legendTable->readOnlyText()) {
      penNumber->setValue(legend.count());
      _legend.setTexts(legend);
    }
    _legend.setPens(legend);
    _legend.setSymbols(legend);
    _legend.setTextColors(legend);
    legendTable->setLegend(&_legend);
    legendTable->setConstant(p.constant());
    return true;
  } else {
    return false;
  }
}

void LegendProperties::on_loadLegend_clicked()
{
  TRACE;
  static QString title=tr("Open an existing legend");
  QString fileName=Message::getOpenFileName(title, tr("Legend attributes (*.legend)"));
  if(!fileName.isEmpty()) {
    XMLErrorReport xmler(XMLErrorReport::Read);
    xmler.setTitle(title);
    xmler.setFileName(fileName);
    XMLSciFigs s;
    SciFigs::Legend legend;
    if(xmler.exec(s.restoreFile(fileName, &legend, XMLSciFigs::Data))) {
      if( !legendTable->readOnlyText()) {
        penNumber->setValue(legend.count());
        _legend.setTexts(legend);
      }
      _legend.setTextColors(legend);
      _legend.setPens(legend);
      _legend.setSymbols(legend);
      legendTable->setLegend(&_legend);
    }
  }
}

void LegendProperties::on_saveLegend_clicked()
{
  TRACE;
  static QString title=tr("Save legend");
  QString fileName=Message::getSaveFileName(title, tr("Legend attributes (*.legend)"));
  if(fileName.length()>0) {
    XMLErrorReport xmler(XMLErrorReport::Write);
    xmler.setTitle(title);
    xmler.setFileName(fileName);
    XMLSciFigs s;
    xmler.exec(s.saveFile(fileName, &_legend, XMLSciFigs::Data));
  }
}

void LegendProperties::on_penNumber_valueChanged(int)
{
  TRACE;
  _legend.resize(penNumber->value());
  legendTable->setLegend(&_legend);
}

void LegendProperties::on_oneColor_toggled(bool)
{
  TRACE;
  legendTable->setOneColor(oneColor->isChecked());
}

void LegendProperties::setReadOnlyText(bool ro)
{
  TRACE;
  penNumber->setEnabled(false);
  legendTable->setReadOnlyText(ro);
}

void LegendProperties::setLegend(const SciFigs::Legend& legend)
{
  TRACE;
  penNumber->setMinimum(1);
  penNumber->setMaximum(1024);
  penNumber->setValue(legend.count());

  _legend=legend;
  legendTable->setLegend(&_legend);
}

void LegendProperties::setPropertySections(LegendTable::PropertySections s)
{
  TRACE;
  legendTable->setPropertySections(s);
}

} // namespace SciFigs
