/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-02-17
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef GRAPHCONTENTLAYER_H
#define GRAPHCONTENTLAYER_H

#include <QGpGuiTools.h>
#include <QGpCoreMath.h>

#include "SciFigsDLLExport.h"

namespace SciFigs {

class GraphContent;
class AxisWindow;
class GraphContentProperties;
class LayerMouseTracking;
class LayerPainterRequest;
class GraphContentOptions;

//#define GRAPHCONTENTLAYER_TESTING

class SCIFIGS_EXPORT GraphContentLayer :  public QObject, public PropertyItem, public XMLClass
{
  Q_OBJECT
  // Overload to mark the beginning of properties to save to XML
  Q_PROPERTY(QString objectName READ objectName WRITE setObjectName STORED true)
  // Kept only to avoid warning on loading old files
  Q_PROPERTY(QString name READ dummyPropertyString WRITE setObjectName STORED false)

  Q_PROPERTY(qreal opacity READ opacity WRITE setOpacity STORED true)
  Q_PROPERTY(QString comments READ comments WRITE setComments STORED true)
  DUMMY_PROPERTIES
public:
  GraphContentLayer(AxisWindow * parent=nullptr);

  virtual const QString& xml_tagName() const {return xmlGraphContentLayerTag;}
  static const QString xmlGraphContentLayerTag;

  void lock() {_mutex.lock();}
  void lockDelayPainting();
  void unlock() {_mutex.unlock();}

  virtual void setParent(GraphContent * parent);
  // Returns the graphcontent parent
  GraphContent * graphContent() {return reinterpret_cast<GraphContent *>(parent());}
  const GraphContent * graphContent() const {return reinterpret_cast<const GraphContent *>(parent());}
  // Returns the axiswindow parent
  AxisWindow * graph() const {return reinterpret_cast<AxisWindow *>(parent()->parent());}

  // From user changes, update the properties
  virtual bool mousePressEvent (QMouseEvent * e, int id=-1) {Q_UNUSED(e) Q_UNUSED(id) return true;}
  virtual bool mouseReleaseEvent (QMouseEvent * e, int id=-1) {Q_UNUSED(e) Q_UNUSED(id) return true;}
  virtual void mouseMoveEvent (const QPoint& pt, Qt::MouseButtons buttons, Qt::KeyboardModifiers modifiers) {
    Q_UNUSED(pt)
    Q_UNUSED(buttons)
    Q_UNUSED(modifiers)
  }
  virtual bool wheelEvent(QWheelEvent * e) {Q_UNUSED(e) return true;}
  virtual bool keyPressEvent(QKeyEvent* e) {Q_UNUSED(e) return true;}
  virtual bool keyReleaseEvent(QKeyEvent* e) {Q_UNUSED(e) return true;}
  virtual inline bool trackRectangle(int id, double rx1,double ry1, double rx2, double ry2, Qt::KeyboardModifiers m);
  void paint(const LayerPainterRequest& lp, QPainter& p, double dotpercm);
  virtual inline void paintText(const LayerPainterRequest& lp, double dotpercm);
  virtual Rect boundingRect() const=0;
  virtual QString coordinateTipInfo(const Point2D& pUser, const Point2D& pReal) const;

  void setOpacity(qreal o);
  qreal opacity() const {return _opacity;}

  void setComments(const QString& c) {_comments=c;}
  const QString& comments() const {return _comments;}

  void addReference() {_referenceCount++;}
  void removeReference() {_referenceCount--; ASSERT(_referenceCount>=0);}
  bool isRemovable() const {return _referenceCount==0;}

  void setEditable(bool r) {_isEditable=r;}
  bool isEditable() const {return _isEditable;}

  void setSelected(bool s) {_isSelected=s;}
  bool isSelected() const {return _isSelected;}

  QAction * addTrackingAction(QString text, int id=0, QString statusTip=QString());
  void setTrackingAction(int id, bool checked);
  static QString trackingActionCtrlHelp();

  virtual bool hasProperties() {return false;}
  virtual void addProperties(PropertyProxy * /* pp */ ) {}
  virtual void removeProperties(PropertyProxy * /* pp */ ) {}
  virtual void properties(PropertyWidget * /* w */ ) const {}
  virtual void setProperty(uint /* wid */, int /* pid */, QVariant /* val */ ) {}

  virtual void xml_layerPolish(XML_POLISH_ARGS) {Q_UNUSED(context)}
public slots:
  void deepUpdate();
  virtual void toggleTrackingAction(bool checked, int id=-1);
signals:
  void dataPainted(QPainter& p, double dotpercm, int w, int h) const;
protected:
  virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
  virtual bool xml_setProperty(XML_SETPROPERTY_ARGS) {return qobject_setProperty(this, memberID, tag, attributes, content, context);}
#ifdef GRAPHCONTENTLAYER_TESTING
  virtual void paintData(const LayerPainterRequest& lp, QPainter& p, double dotpercm) const;
#else
  virtual void paintData(const LayerPainterRequest& lp, QPainter& p, double dotpercm) const=0;
#endif
  void beginMouseTracking(const LayerMouseTracking& mt);
  void endMouseTracking(int id=0);
  bool isMouseTracking(int id=0) const;

  qreal _opacity;
  int _referenceCount;
  uint _isEditable:1;
  uint _isSelected:1;
  QString _comments;

  // Avoid a layer to be painted at the same time by two distinct threads
  mutable Mutex _mutex;
};

inline bool GraphContentLayer::trackRectangle(int id, double rx1,double ry1, double rx2, double ry2, Qt::KeyboardModifiers m)
{
  TRACE;
  Q_UNUSED(id)
  Q_UNUSED(rx1)
  Q_UNUSED(ry1)
  Q_UNUSED(rx2)
  Q_UNUSED(ry2)
  Q_UNUSED(m)
  return false;
}

inline void GraphContentLayer::paintText(const LayerPainterRequest& lp, double dotpercm)
{
  TRACE;
  Q_UNUSED(lp)
  Q_UNUSED(dotpercm)
}

} // namespace SciFigs

#endif // GRAPHCONTENTLAYER_H
