/***************************************************************************
**
**  This file is part of QGpGuiTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2011-01-03
**  Copyright: 2011-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include <QtNetwork>

#include "HttpProxy.h"

namespace QGpGuiTools {

  /*!
    \class HttpProxy HttpProxy.h
    \brief Brief description of class still missing

    Avoids exposure of QtNetwork in header.
  */

  /*!
    No proxy
  */
  HttpProxy::HttpProxy()
  {
    TRACE;
    _proxy=new QNetworkProxy(QNetworkProxy::NoProxy);
  }

  /*!
    Proxy defined by \a proxy
  */
  HttpProxy::HttpProxy(const QNetworkProxy& proxy)
  {
    TRACE;
    _proxy=new QNetworkProxy(proxy);
  }

  /*!
    Proxy defined by an \a url: [http://][username:password@]address:port
  */
  HttpProxy::HttpProxy(QString url)
  {
    TRACE;
    if(!url.startsWith("http://")) {
      url.prepend("http://");
    }
    QUrl proxy(url);
    _proxy=new QNetworkProxy;
    _proxy->setHostName(proxy.host());
    _proxy->setPort(proxy.port());
    _proxy->setUser(proxy.userName());
    _proxy->setPassword(proxy.password());
    _proxy->setType(QNetworkProxy::HttpProxy);
  }

  /*!
    Copy constructor.
  */
  HttpProxy::HttpProxy(const HttpProxy& o)
  {
    TRACE;
    _proxy=new QNetworkProxy(*o._proxy);
  }

  /*!
    Description of destructor still missing
  */
  HttpProxy::~HttpProxy()
  {
    TRACE;
    delete _proxy;
  }

  /*!
    Assignment operator
  */
  void HttpProxy::operator=(const HttpProxy& o)
  {
    TRACE;
    delete _proxy;
    _proxy=new QNetworkProxy(*o._proxy);
  }

  /*!
    Comparison operator
  */
  bool HttpProxy::operator==(const HttpProxy& o) const
  {
    TRACE;
    return *_proxy==*o._proxy;
  }

  /*!
    Comparison operator
  */
  bool HttpProxy::operator<(const HttpProxy& o) const
  {
    TRACE;
    return compareProxies(*_proxy, *o._proxy)<0;
  }

  /*!
    QNetworkProxy has no operator<() or toString(). A comparison function is thus necessary.
    Returns -1 if \a p1 is less than \a p2, 1 for greater and 0 for equal. The order of prioirity
    is type, hostName, port, user, password
  */
  int HttpProxy::compareProxies(const QNetworkProxy& p1, const QNetworkProxy& p2)
  {
    TRACE;
    if(p1.type()<p2.type()) {
       return -1;
    } else if(p1.type()>p2.type()){
      return 1;
    } else {
      if(p1.hostName()<p2.hostName()) {
        return -1;
      } else if(p1.hostName()>p2.hostName()) {
        return 1;
      } else {
        if(p1.port()<p2.port()) {
          return -1;
        } else if(p1.port()>p2.port()) {
          return 1;
        } else {
          if(p1.user()<p2.user()) {
            return -1;
          } else if(p1.user()>p2.user()) {
            return 1;
          } else {
            if(p1.password()<p2.password()) {
              return -1;
            } else if(p1.password()>p2.password()) {
              return 1;
            } else {
              return 0;
            }
          }
        }
      }
    }
  }

  /*!
    For debug, prints proxy data
  */
  void HttpProxy::printDebug(QTextStream& s) const
  {
    TRACE;
    switch(_proxy->type()) {
    case QNetworkProxy::NoProxy:
      s << "  Direct access\n";
      break;
    default:
      if(_proxy->user().isEmpty()) {
        s << "  " << _proxy->hostName()
          << ":" << _proxy->port()
          << "(type " << _proxy->type() << ")\n";
      } else {
        s << "  " << _proxy->user()
          << ":" << _proxy->password()
          << "@" << _proxy->hostName()
          << ":" << _proxy->port()
          << "(type " << _proxy->type() << ")\n";
      }
      break;
    }
  }

  /*!
    Returns the proxy
  */
  const QNetworkProxy& HttpProxy::proxy() const
  {
    TRACE;
    return *_proxy;
  }

} // namespace QGpGuiTools
