/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-09-26
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef MODE_H
#define MODE_H

#include <QGpCoreTools.h>
#include "QGpCoreWaveDLLExport.h"

namespace QGpCoreWave {

  class QGPCOREWAVE_EXPORT Mode : public XMLClass
  {
    TRANSLATIONS("Mode")
  public:
    enum Type {Dispersion, Ellipticity, Autocorr};
    enum Polarization {Vertical=1, Radial=2, Rayleigh=3, Transverse=4, Love=5};
    enum Slowness {Phase=0, Group=1};
    enum Value {Signed=0, Absolute=1};

    inline Mode();
    inline Mode(Slowness s, Polarization p, int index);
    inline Mode(Value v, int index);
    inline Mode(Polarization p, int ringIndex, int index);
    inline Mode(const Mode& o);

    virtual const QString& xml_tagName() const {return xmlModeTag;}
    static const QString xmlModeTag;

    bool operator==(const Mode& o) const;

    void setPolarization(Polarization p) {_polarization=p;}
    Polarization polarization() const {return _polarization;}

    static QString polarization(Polarization p);
    static Polarization polarization(QString p);
    static QString userPolarization(Polarization p);
    static Polarization userPolarization(QString p);

    void setSlowness(Slowness s) {_slowness=s;}
    Slowness slowness() const {return _slowness;}

    static QString slowness(Slowness s);
    static Slowness slowness(QString s);
    static QString userSlowness(Slowness s);
    static Slowness userSlowness(QString s);

    void setValue(Value v) {_value=v;}
    Value value() const {return _value;}

    static QString value(Value s);
    static Value value(QString s);
    static QString userValue(Value s);
    static Value userValue(QString s);

    void setRingIndex(int i) {_ringIndex=i;}
    int ringIndex() const {return _ringIndex;}

    void setIndex(int i) {_index=i;}
    int index() const {return _index;}
    QString indexString() const;
    QString toString(Type t) const;
  protected:
    void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
    XMLMember xml_member(XML_MEMBER_ARGS);
    bool xml_setProperty(XML_SETPROPERTY_ARGS);
  private:
    Value _value;
    Slowness _slowness;
    Polarization _polarization;
    int _ringIndex;
    int _index;
  };

  inline Mode::Mode()
  {
    TRACE;
    _value=Signed;
    _polarization=Rayleigh;
    _slowness=Phase;
    _ringIndex=0;
    _index=0;
  }

  inline Mode::Mode(Polarization p, int ringIndex, int index)
  {
    TRACE;
    _value=Signed;
    _polarization=p;
    _slowness=Phase;
    _ringIndex=ringIndex;
    _index=index;
  }

  inline Mode::Mode(Value v, int index)
  {
    TRACE;
    _value=v;
    _polarization=Rayleigh;
    _slowness=Phase;
    _ringIndex=0;
    _index=index;
  }

  inline Mode::Mode(Slowness s, Polarization p, int index)
  {
    TRACE;
    _value=Signed;
    _slowness=s;
    _polarization=p;
    _ringIndex=0;
    _index=index;
  }

  inline Mode::Mode(const Mode& o) : XMLClass()
  {
    TRACE;
    _value=o._value;
    _slowness=o._slowness;
    _polarization=o._polarization;
    _ringIndex=o._ringIndex;
    _index=o._index;
  }

  inline bool Mode::operator==(const Mode& o) const
  {
    TRACE;
    return _value==o._value &&
           _slowness==o._slowness &&
           _polarization==o._polarization &&
           _ringIndex==o._ringIndex &&
           _index==o._index;
  }

} // namespace QGpCoreWave

#endif // MODE_H
