/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-09-26
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "Mode.h"

namespace QGpCoreWave {

  const QString Mode::xmlModeTag="Mode";

  /*!
    \class Mode Mode.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Return a standard string for slowness \a s.
  */
  QString Mode::slowness(Slowness s)
  {
    TRACE;
    switch (s) {
    case Group:
      return "Group";
    case Phase:
      break;
    }
    return "Phase";
  }

  /*!
    Return slowness from a standard string \a s
  */
  Mode::Slowness Mode::slowness(QString s)
  {
    TRACE;
    if(s=="Group") {
      return Group;
    } else {
      return Phase;
    }
  }

  /*!
    Return a user-friendly string for slowness \a s.
  */
  QString Mode::userSlowness(Slowness s)
  {
    TRACE;
    switch (s) {
    case Group:
      return tr("Group");
    case Phase:
      break;
    }
    return tr("Phase");
  }

  /*!
    Return slowness from a user-friendly string \a s
  */
  Mode::Slowness Mode::userSlowness(QString s)
  {
    TRACE;
    if(s==tr("Group")) {
      return Group;
    } else {
      return Phase;
    }
  }

  /*!
    Return a standard string for polarization \a p.
  */
  QString Mode::polarization(Polarization p)
  {
    TRACE;
    switch (p) {
    case Vertical:
      break;
    case Radial:
      return "Radial";
    case Rayleigh:
      return "Rayleigh";
    case Transverse:
      return "Transverse";
    case Love:
      return "Love";
    }
    return "Vertical";
  }

  /*!
    Return polarization from a standard string \a p.
  */
  Mode::Polarization Mode::polarization(QString p)
  {
    TRACE;
    if(p.count()>2) {
      switch(p[2].unicode()) {
      case 'd':
        return Radial;
      case 'a':
        return Transverse;
      case 'v':
        return Love;
      case 'r':
        return Vertical;
      default:
        break;
      }
    }
    return Rayleigh;
  }

  /*!
    Return a user-friendly string for polarization \a p.
  */
  QString Mode::userPolarization(Polarization p)
  {
    TRACE;
    switch (p) {
    case Vertical:
      break;
    case Radial:
      return tr("Radial");
    case Rayleigh:
      return tr("Rayleigh");
    case Transverse:
      return tr("Transverse");
    case Love:
      return tr("Love");
    }
    return tr("Vertical");
  }

  /*!
    Return polarization from a user-friendly string \a p.
  */
  Mode::Polarization Mode::userPolarization(QString p)
  {
    TRACE;
    if(p==tr("Radial")) {
      return Radial;
    } else if(p==tr("Transverse")) {
      return Transverse;
    } else if(p==tr("Love")) {
      return Love;
    } else if(p==tr("Vertical")) {
      return Vertical;
    } else {
      return Rayleigh;
    }
  }


  /*!
    Return a standard string for slowness \a s.
  */
  QString Mode::value(Value s)
  {
    TRACE;
    switch (s) {
    case Signed:
      break;
    case Absolute:
      return "Absolute";
    }
    return "Signed";
  }

  /*!
    Return slowness from a standard string \a s
  */
  Mode::Value Mode::value(QString s)
  {
    TRACE;
    if(s=="Absolute") {
      return Absolute;
    } else {
      return Signed;
    }
  }

  /*!
    Return a user-friendly string for slowness \a s.
  */
  QString Mode::userValue(Value s)
  {
    TRACE;
    switch (s) {
    case Absolute:
      return tr("Absolute");
    default:
      break;
    }
    return tr("Signed");
  }

  /*!
    Return slowness from a user-friendly string \a s
  */
  Mode::Value Mode::userValue(QString s)
  {
    TRACE;
    if(s==tr("Absolute")) {
      return Absolute;
    } else {
      return Signed;
    }
  }

  /*!
    Return a user-friendly string describing this mode index
  */
  QString Mode::indexString() const
  {
    TRACE;
    switch (_index) {
    case 0:
      return tr("fundamental mode");
    case 1:
      return tr("1st higher mode");
    case 2:
      return tr("2nd higher mode");
    case 3:
      return tr("3rd higher mode");
    default:
      return tr("higher mode %1").arg(_index);
    }
  }

  /*!
    Return a user-friendly string completely describing this mode
  */
  QString Mode::toString(Type t) const
  {
    TRACE;
    QString str;
    switch(t) {
    case Dispersion:
      str+=userSlowness(_slowness);
      str+=" ";
      str+=userPolarization(_polarization);
      break;
    case Ellipticity:
      str+=userValue(_value);
      break;
    case Autocorr:
      str+=tr("ring ");
      str+=QString::number(_ringIndex);
      str+=userPolarization(_polarization);
      break;
    }
    str+=" ";
    str+=indexString();
    return str;
  }

  void Mode::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
  {
    TRACE;
    Q_UNUSED(context)
    writeProperty(s,"value", value(_value));
    writeProperty(s,"slowness", slowness(_slowness));
    writeProperty(s,"polarization", polarization(_polarization));
    writeProperty(s,"ringIndex", _ringIndex);
    writeProperty(s,"index",_index);
  }

  XMLMember Mode::xml_member(XML_MEMBER_ARGS)
  {
    TRACE;
    Q_UNUSED(attributes)
    Q_UNUSED(context)
    if(tag=="slowness") return XMLMember(0);
    else if(tag=="polarization") return XMLMember(1);
    else if(tag=="ringIndex") return XMLMember(2);
    else if(tag=="index") return XMLMember(3);
    else if(tag=="value") return XMLMember(4);
    else if(tag=="polarisation") return XMLMember(1);            // Kept for compatibility
    else return XMLMember(XMLMember::Unknown);
  }

  bool Mode::xml_setProperty(XML_SETPROPERTY_ARGS)
  {
    TRACE;
    Q_UNUSED(tag)
    Q_UNUSED(attributes)
    Q_UNUSED(context)
    switch (memberID) {
    case 0:
      _slowness=slowness(content.toString());
      return true;
    case 1:
      _polarization=polarization(content.toString());
      return true;
    case 2:
      _ringIndex=content.toInt();
      return true;
    case 3:
      _index=content.toInt();
      return true;
    case 4:
      _value=value(content.toString());
      return true;
    default:
      return false;
    }
  }

} // namespace QGpCoreWave
