/***************************************************************************
**
**  This file is part of QGpCompatibility.
**
**  QGpCompatibility is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCompatibility is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-05-19
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>
#include "CompatMultiModalFrequency.h"

namespace QGpCompatibility {

CompatMultiModalFrequency::CompatMultiModalFrequency()
{
  TRACE;
  _omegas=0;
  _omegasCount=0;
  _modesCount=0;
  _freq2omega=2*M_PI;
}

CompatMultiModalFrequency::CompatMultiModalFrequency(int modesCount, int omegasCount)
{
  TRACE;
  _modesCount=modesCount;
  _omegasCount=omegasCount;
  // Allocates memory for omegas
  _omegas=new double[_omegasCount];
  _freq2omega=2*M_PI;
}

CompatMultiModalFrequency::CompatMultiModalFrequency(const CompatMultiModalFrequency * o)
{
  TRACE;
  _modesCount=o->modesCount();
  _omegasCount=o->omegasCount();
  // Allocates memory for omegas
  _omegas=new double[_omegasCount];
  // Copy omegas
  for(int i=0;i<_omegasCount;i++)  _omegas[i]=o->omega(i);
  _freq2omega=2*M_PI;
}

CompatMultiModalFrequency::CompatMultiModalFrequency (const CompatMultiModalFrequency * o,
                                                int modesCount)
{
  TRACE;
  _modesCount=modesCount;
  _omegasCount=o->_omegasCount;
  // Allocates memory for omegas
  _omegas=new double[_omegasCount];
  // Copy omegas
  for(int i=0;i<_omegasCount;i++) _omegas[i]=o->omega(i);
  _freq2omega=2*M_PI;
}

CompatMultiModalFrequency::~CompatMultiModalFrequency()
{
  TRACE;
  delete [] _omegas;
}

void CompatMultiModalFrequency::setFrequencies(QVector<double> * values)
{
  TRACE;
  ASSERT (_omegasCount==(int)values->size());
  for(int i=0;i<_omegasCount;i++) _omegas[i]=_freq2omega*(*values)[i];
  ASSERT (_omegas[0]<_omegas[_omegasCount-1]);
}

void CompatMultiModalFrequency::setPeriods(QVector<double> * values)
{
  TRACE;
  ASSERT (_omegasCount==(int)values->size());
  for(int i=0;i<_omegasCount;i++) _omegas[_omegasCount-i-1]=_freq2omega/(*values)[i];
  ASSERT (_omegas[0]<_omegas[_omegasCount-1]);
}

bool CompatMultiModalFrequency::isSameOmegas(const  CompatMultiModalFrequency* o) const
{
  TRACE;
  if(_omegasCount!=o->_omegasCount) return false;
  for(int i=0;i<_omegasCount;i++) {
    if(_omegas[i]!=o->_omegas[i]) return false;
  }
  return true;
}

void CompatMultiModalFrequency::omegaToReport(QDataStream& s) const
{
  TRACE;
  s << _modesCount;
  s << _omegasCount;  
  for(int i=0;i<_omegasCount;i++) s << _omegas[i];
}

void CompatMultiModalFrequency::reportToOmega(QDataStream& s)
{
  TRACE;
  s >> _modesCount;
  s >> _omegasCount;
  delete [] _omegas;
  _omegas=new double[_omegasCount];
  for(int i=0;i<_omegasCount;i++) s >> _omegas[i];
}

QVector<double> * CompatMultiModalFrequency::periods()
{
  QVector<double> * list=new QVector<double>;
  list->reserve(_omegasCount);
  for(int i=0;i<_omegasCount;i++)
    list->push_back(_freq2omega/_omegas[i] );
  return list;
}

QVector<double> * CompatMultiModalFrequency::frequencies()
{
  QVector<double> * list=new QVector<double>;
  list->reserve(_omegasCount);
  for(int i=0;i<_omegasCount;i++)
    list->push_back(_omegas[i]/_freq2omega);
  return list;
}

#if 0
void CompatMultiModalFrequency::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
{
  TRACE;
  writeProperty(s, "modesCount",_modesCount);
  writeProperty(s, "omegasCount",_omegasCount);
  s.printf("%s<property name=\"%s\">\n",indent,"omegas");
  for(int i=0;i<_omegasCount;i++)
    s.printf("%s    %lg\n",indent,_omegas[i]/_freq2omega);
  s.printf("%s</property>\n",indent);
}

bool CompatMultiModalFrequency::xml_setProperty(CompatStringSection& propertyName,
                                             CompatStringSection& content)
{
  TRACE;
  if(propertyName=="modesCount") _modesCount=content.toInt();
  else if(propertyName=="omegasCount") {
    _omegasCount=content.toInt();
    delete [] _omegas;
    _omegas=new double[_omegasCount];
  }
  else if(propertyName=="omegas") {
    const char * ptr=0;
    for(int i=0;i<_omegasCount;i++) {
      CompatStringSection field=content.nextField(ptr);
      if(field.isValid()) _omegas[i]=_freq2omega*field.toDouble();
      else _omegas[i]=0;
    }
  }
  else return false;
  return true;  
}
#endif

} // namespace QGpCompatibility
