/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-10-20
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef SIGNALFILEFORMAT_H
#define SIGNALFILEFORMAT_H

#include <QGpCoreTools.h>

#include "GeopsyCoreDLLExport.h"

namespace GeopsyCore {

class AbstractFileFormat;

class GEOPSYCORE_EXPORT SignalFileFormat
{
public:
  // Be careful, for the sake of compatibility with older databases, never
  // change the order of the following enumeration. Only append. This is not valid for new db.
  // 2018-11-08: this compatibility with very old databases is now dropped after the removal
  //             of NiSismo format (unknow equipment and manufacturer, no result in a web engine)
  enum Format {
    Unknown=0,
    Seg2,
    SuLittleEndian,
    SuBigEndian,
    Tomo,
    RD3,
    SacLittleEndian,
    SacBigEndian,
    Radan,
    Gse2,
    CityShark2,
    MiniShark,
    Ascii,
    GeopsySignal,
    Saf,
    MultiGse2,
    Sismalp,
    Wav,
    SegYLittleEndian,
    SegYBigEndian,
    Temporary,
    SyscomXmr,
    SyscomSmr,
    SyscomVmrx,
    GuralpGcf,
    MiniSeed,
    AsciiOneColumn,
    PasscalSegYBigEndian,
    PasscalSegYLittleEndian,
    SegD,
    Custom,
    Fourier,
    FormatCount  // Must be the last one, used to count the available file formats
  };
  enum Storage {Single, Multi};

  SignalFileFormat(Format f=Unknown, AbstractFileFormat * customFormat=nullptr);
  SignalFileFormat(const SignalFileFormat& o);
  ~SignalFileFormat();

  void operator=(const SignalFileFormat& o);
  bool operator==(const SignalFileFormat& o) const;
  bool operator<(const SignalFileFormat& o) const;

  Format id() const {return _id;}
  const AbstractFileFormat * customFormat() const {return _customFormat;}
  bool isReadOnly() const;
  Storage storage() const;

  QString name() const;
  QString caption() const;
  QString captionFilter() const;
  QString filter() const;
  QString suffix() const;

  static SignalFileFormat fromContent(QString fileName);
  static SignalFileFormat fromSuffix(QString suffix);
  static SignalFileFormat fromName(QString n);
  static SignalFileFormat fromCaptionFilter(QString cf);
  static QList< QPair<QString, SignalFileFormat> > importList();
  static QList< QPair<QString, SignalFileFormat> > exportList();
private:
  Format _id;
  AbstractFileFormat * _customFormat;
};

#define SIGNALFILEFORMAT_READONLY \
  case SignalFileFormat::RD3: \
  case SignalFileFormat::Radan: \
  case SignalFileFormat::MiniShark: \
  case SignalFileFormat::Sismalp: \
  case SignalFileFormat::PasscalSegYLittleEndian: \
  case SignalFileFormat::PasscalSegYBigEndian: \
  case SignalFileFormat::Temporary: \
  case SignalFileFormat::SyscomXmr: \
  case SignalFileFormat::SyscomSmr: \
  case SignalFileFormat::SyscomVmrx: \
  case SignalFileFormat::GuralpGcf: \
  case SignalFileFormat::SegD: \
  case SignalFileFormat::Fourier: \
  case SignalFileFormat::FormatCount: \
  case SignalFileFormat::Unknown:

} // namespace GeopsyCore

#endif // SIGNALFILEFORMAT_H
