/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-11-10
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/
#ifndef SIGNALFILE_H
#define SIGNALFILE_H

#include <QGpCoreTools.h>

#include "GeopsyCoreDLLExport.h"
#include "SignalFileFormat.h"

namespace GeopsyCore {

class Signal;
class SignalDatabase;
class SubSignalPool;
class GeopsySignalHeader;
class SignalFileFormat;

class GEOPSYCORE_EXPORT SignalFile : public XMLClass
{
  TRANSLATIONS("SignalFile")
public:
  SignalFile(SignalDatabase * db,
             QString fileName=QString(),
             const SignalFileFormat& format=SignalFileFormat::Unknown);
  SignalFile(SignalDatabase * db, SignalFile& f, QDir& d);
  ~SignalFile();

  virtual const QString& xml_tagName() const {return xmlSignalFileTag;}
  static const QString xmlSignalFileTag;

  SignalFile& operator=(const SignalFile& p);
  QString name(int len) const;
  QString shortName() const;
  // Interface function to data members
  QString name() const {return _name;}
  void setName(QString n) {_name=n;}

  qint64 size() const {return _size;}
  quint32 crc32() const {return _crc32;}
  QString keyName() const;

  const SignalFileFormat& format() const {return _format;}
  void setFormat(const SignalFileFormat& f) {_format=f;}
  void setFormat(QString f) {_format=SignalFileFormat::fromName(f);}

  bool isOriginalFile() {return _isOriginalFile;}
  void setOriginalFile(bool b) {_isOriginalFile=b;}

  bool isPathModified() {return _pathModified;}
  void setPathModified(bool b) {_pathModified=b;}

  SignalDatabase * database() const {return _db;}
  void setDatabase(SignalDatabase * db) {_db=db;}

  bool hasValidContent(const QString& fileName) const;
  void removeFile();
  bool load();
  bool loadGeopsySignal(const SubSignalPool& subPool);

  static void setSignalName(Signal * sig, QString stationName, QString sufix,
                             int recNum, QString fileName);
  bool save(const QDir& d);
protected:
  virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  virtual void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
  virtual bool xml_setProperty(XML_SETPROPERTY_ARGS);
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
  virtual bool xml_polish(XML_POLISH_ARGS);
private:
  SignalFileFormat _format;
  QString _name;
  bool _isOriginalFile;
  bool _pathModified;
  quint32 _crc32;
  qint64 _size;
  SignalDatabase * _db;

  bool setContentKey();
  bool loadGeopsySignal();
  bool loadShark(SignalFileFormat format);
  bool loadSeg2();
  bool loadSegD();
  bool loadSu(QDataStream::ByteOrder bo);
  bool loadSegY(QDataStream::ByteOrder bo);
  bool loadPasscalSegY(QDataStream::ByteOrder bo);
  bool loadRD3();
  bool loadSac(QDataStream::ByteOrder bo);
  bool loadRadan();
  bool loadGse2(SignalFileFormat::Storage sto=SignalFileFormat::Multi);
  bool loadAsciiOneColumn();
  bool loadAscii();
  bool loadFourier();
  bool loadSaf();
  bool loadSismalp();
  bool loadWav();
  bool loadSyscomXmr();
  bool loadSyscomSmr();
  bool loadSyscomVmrx();
  bool loadGuralpGcf();
  bool loadMiniSeed();

  void warnAsciiHeader();
};

} // namespace GeopsyCore

#endif // SIGNALFILE_H
