/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2011-05-18
**  Copyright: 2011-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "MasterSignalGroup.h"
#include "DefaultSignalGroup.h"
#include "SignalGroupFolder.h"

namespace GeopsyCore {

  /*!
    \class MasterSignalGroup MasterSignalGroup.h
    \brief Brief description of class still missing

    It is the group at the top of the group tree. It is a read only group which children
    are not read-only by default. It inherits QObject to forward signals about is contents
    to other objects of the application.
  */

  const QString MasterSignalGroup::xmlMasterSignalGroupTag="MasterGroup";

  /*!
    Master signal group has no parent.
  */
  MasterSignalGroup::MasterSignalGroup(SignalDatabase * database)
    : QObject(), AbstractSignalGroup()
  {
    TRACE;
    SignalGroupFolder * defaultFolder;
    defaultFolder=new SignalGroupFolder(this);
    defaultFolder->setName(tr("Default groups"));
    defaultFolder->setComments(tr("Some default groups to acces signals"));
    defaultFolder->setReadOnly(true);
    defaultFolder->setReadOnlyChildren(true);
    defaultFolder->setStored(false);
    new AllSignalsGroup(database, defaultFolder);
    new TemporarySignalsGroup(database, defaultFolder);
    new AllFilesGroup(database, defaultFolder);
    new TemporaryFilesGroup(database, defaultFolder);
    new PermanentFilesGroup(database, defaultFolder);
    SignalGroupFolder * byFolder;
    byFolder=new SignalGroupFolder(defaultFolder);
    byFolder->setName(tr("By names"));
    byFolder->setComments(tr("Not yet available"));
    byFolder->setReadOnly(true);
    byFolder->setReadOnlyChildren(true);
    byFolder=new SignalGroupFolder(defaultFolder);
    byFolder->setName(tr("By components"));
    byFolder->setComments(tr("Not yet available"));
    byFolder->setReadOnly(true);
    byFolder->setReadOnlyChildren(true);
    byFolder=new SignalGroupFolder(defaultFolder);
    byFolder->setName(tr("By receivers"));
    byFolder->setComments(tr("Not yet available"));
    byFolder->setReadOnly(true);
    byFolder->setReadOnlyChildren(true);
    byFolder=new SignalGroupFolder(defaultFolder);
    byFolder->setName(tr("By sources"));
    byFolder->setComments(tr("Not yet available"));
    byFolder->setReadOnly(true);
    byFolder->setReadOnlyChildren(true);
  }

  /*!
    Deletes all children groups except the default one
  */
  void MasterSignalGroup::clear()
  {
    TRACE;
    int i=0;
    while(i<childrenCount()) {
      AbstractSignalGroup * g=static_cast<AbstractSignalGroup *>(childAt(i));
      if(!g->readOnly()) {
        emit parentAboutToBeChanged(g, 0);
        g->setParent(0);
        emit parentChanged(g, this);
        delete g;
      } else {
        i++;
      }
    }
    setModified(false);
  }

  /*!
    In normal situation this function should always return 0 or 1. If an ID is duplicated, it returns a value > 1.
  */
  int MasterSignalGroup::countId(int id, int& newId) const
  {
    TRACE;
    return AbstractSignalGroup::countId(id, newId);
  }

  bool MasterSignalGroup::xml_inherits(const QString& tagName) const
  {
    if(tagName==xmlMasterSignalGroupTag) {
      return true;
    } else {
      return AbstractSignalGroup::xml_inherits(tagName);
    }
  }

  /*!
    Avoids 'name' and 'comments', always set to null or simply ignored.
  */
  void MasterSignalGroup::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
  {
    TRACE;
    Q_UNUSED(s)
    Q_UNUSED(context)
  }

  XMLMember MasterSignalGroup::xml_member(XML_MEMBER_ARGS)
  {
    TRACE;
    if(tag.size()>0) {
      switch (tag[0].unicode()) {
      case 'c':
        if(tag=="comments") return XMLMember(XMLMember::Skip);  // Kept for compatibility
        break;
      case 'n':
        if(tag=="name") return XMLMember(XMLMember::Skip);      // Kept for compatibility
        break;
      case 'f':
        if(tag=="folder") return XMLMember(XMLMember::Skip);    // Kept for compatibility
        break;
      default:
        break;
      }
    }
    return AbstractSignalGroup::xml_member(tag, attributes, context);
  }

  /*!
    Sends a Qt signal when a (great-)children is reported as modified.
  */
  void MasterSignalGroup::reportParentAboutToBeChanged(AbstractSignalGroup * g, AbstractSignalGroup * newParent) const
  {
    emit parentAboutToBeChanged(g, newParent);
  }

  /*!
    Sends a Qt signal when a (great-)children is reported as modified.
  */
  void MasterSignalGroup::reportParentChanged(AbstractSignalGroup * g, AbstractSignalGroup * oldParent) const
  {
    emit parentChanged(g, oldParent);
  }

  /*!
    Sends a Qt signal when a (great-)children is reported as modified.
  */
  void MasterSignalGroup::reportDataChanged(AbstractSignalGroup * g) const
  {
    emit dataChanged(g);
  }

} // namespace GeopsyCore
