/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2011-05-20
**  Copyright: 2011-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "DefaultSignalGroup.h"
#include "SignalDatabase.h"
#include "SignalFilePool.h"

namespace GeopsyCore {

  /*!
    \class DefaultSignalGroup DefaultSignalGroup.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  const QString DefaultSignalGroup::xmlDefaultSignalGroupTag="DefaultGroup";

  DefaultSignalGroup::DefaultSignalGroup(SignalDatabase * database, AbstractSignalGroup * parent)
    : AbstractSignalGroup(parent)
  {
    _database=database;
  }

  bool DefaultSignalGroup::xml_inherits(const QString& tagName) const
  {
    if(tagName==xmlDefaultSignalGroupTag) {
      return true;
    } else {
      return AbstractSignalGroup::xml_inherits(tagName);
    }
  }

  /*!
    Do not save these groups
  */
  void DefaultSignalGroup::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
  {
    TRACE;
    Q_UNUSED(s)
    Q_UNUSED(context)
  }

  /*!
    Do not save these groups
  */
  void DefaultSignalGroup::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
  {
    TRACE;
    Q_UNUSED(s)
    Q_UNUSED(context)
  }

  /*!
    Do not save these groups
  */
  XMLMember DefaultSignalGroup::xml_member(XML_MEMBER_ARGS)
  {
    TRACE;
    Q_UNUSED(attributes)
    Q_UNUSED(tag)
    Q_UNUSED(context)
    return XMLMember(XMLMember::Unknown);
  }

  AllSignalsGroup::AllSignalsGroup(SignalDatabase * database, AbstractSignalGroup * parent)
    : DefaultSignalGroup(database, parent)
  {
    setName(tr("All signals"));
    setComments(tr("A list of all signals (permanent and temporary)"));
  }

  SubSignalPool AllSignalsGroup::subPool() const
  {
    SubSignalPool sel=database()->subPool();
    sel.setName(name());
    return sel;
  }

  TemporarySignalsGroup::TemporarySignalsGroup(SignalDatabase * database, AbstractSignalGroup * parent)
    : DefaultSignalGroup(database, parent)
  {
    setName(tr("Temporary signals"));
    setComments(tr("A list of all temporary signals including temporary files"));
  }

  SubSignalPool TemporarySignalsGroup::subPool() const
  {
    SubSignalPool sel;
    const SubSignalPool& all=database()->subPool();
    for(SubSignalPool::const_iterator it=all.begin(); it!=all.end(); ++it) {
      if(!(*it)->file()) sel.addSignal(*it);
    }
    sel.setName(name());
    return sel;
  }

  AllFilesGroup::AllFilesGroup(SignalDatabase * database, AbstractSignalGroup * parent)
    : DefaultSignalGroup(database, parent)
  {
    setName(tr("All files"));
    setComments(tr("A list of all signals in all files (permanent and temporary), "
                   "excludes temporary signals"));
  }

  SubSignalPool AllFilesGroup::subPool() const
  {
    SubSignalPool sel;
    const SignalFilePool& f=database()->filePool();
    for(SignalFilePool::const_iterator it=f.begin(); it!=f.end(); ++it) {
      sel.addFile(*it);
    }
    sel.setName(name());
    return sel;
  }

  TemporaryFilesGroup::TemporaryFilesGroup(SignalDatabase * database, AbstractSignalGroup * parent)
    : DefaultSignalGroup(database, parent)
  {
    setName(tr("Temporary files"));
    setComments(tr("A list of all signals in all temporary files"));
  }

  SubSignalPool TemporaryFilesGroup::subPool() const
  {
    SubSignalPool sel;
    const SignalFilePool& all=database()->filePool();
    for(SignalFilePool::const_iterator it=all.begin(); it!=all.end(); ++it) {
      if((*it)->format().id()==SignalFileFormat::Temporary) {
        sel.addFile(*it);
      }
    }
    sel.setName(name());
    return sel;
  }

  PermanentFilesGroup::PermanentFilesGroup(SignalDatabase * database, AbstractSignalGroup * parent)
    : DefaultSignalGroup(database, parent)
  {
    setName(tr("Permanent files"));
    setComments(tr("A list of all signals in all permanent files"));
  }

  SubSignalPool PermanentFilesGroup::subPool() const
  {
    SubSignalPool sel;
    const SignalFilePool& all=database()->filePool();
    for(SignalFilePool::const_iterator it=all.begin(); it!=all.end(); ++it) {
      if((*it)->format().id()!=SignalFileFormat::Temporary) {
        sel.addFile(*it);
      }
    }
    sel.setName(name());
    return sel;
  }

} // namespace GeopsyCore
