/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-01-04
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "OmniFKRayleigh.h"
#include "FKGridSearch.h"
#include "FKCrossSpectrum.h"
#include "FKPower.h"
#include "FKSteeringTwoComponentRayleighRadial.h"

namespace ArrayCore {

  /*!
    \class OmniFKRayleigh OmniFKRayleigh.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  void OmniFKRayleigh::resetCrossSpectrum()
  {
    _crossSpectrum->resetRayleigh();
  }

  void OmniFKRayleigh::addCrossSpectrum()
  {
    _crossSpectrum->addThreeComponent();
  }

  void OmniFKRayleigh::meanCrossSpectrum(int nBlocks)
  {
    _crossSpectrum->meanRayleigh(nBlocks);
  }

  bool OmniFKRayleigh::invertCrossSpectrum()
  {
    return _crossSpectrum->invertRayleigh();
  }

  void OmniFKRayleigh::setGrid(FKGridSearch * g, double step, double size)
  {
    g->setGrid(0.0, size, step, -0.5*M_PI, 0.5*M_PI, M_PI/20);
    //setAngleGrid();
  }

  AbstractFunction * OmniFKRayleigh::createAngleFunction() const
  {
    return new Angle(_gridCache);
  }

  /*!
    Automatically called by GridSearch upon initialization.
    If returned value is false, grid is assumed to be already initialized.
  */
  bool OmniFKRayleigh::initGrid(int n)
  {
    if(_gridCache->isEmpty(FKCache::TwoComponentRayleigh)) {
      n*=_crossSpectrum->rotationStepCount();
      App::log(tr("Caching Rayleigh steering vectors for 2-component Rayleigh (omni)... (%1 values)\n").arg(n) );
      _gridCache->resize(FKCache::TwoComponentRayleigh, n);
      return true;
    } else {
      return false;
    }
  }

  void OmniFKRayleigh::initGrid(const Point& kell, int index)
  {
    int n=_crossSpectrum->rotationStepCount();
    double da=_crossSpectrum->rotationStep();
    double a;
    Point k;
    k.setZ(kell.y());
    for(int i=0; i<n; i++) {
      a=(double)i*da;
      k.setX(kell.x()*::cos(a));
      k.setY(kell.x()*::sin(a));
      FKSteeringTwoComponentRayleighRadial::init(_gridCache, index*n+i , k);
    }
  }

  double OmniFKRayleigh::value(const Point& kell, int index) const
  {
    double k2=kell.x()*kell.x();
    if(k2>minimumK2() && k2<maximumK2()) {
      int n=_crossSpectrum->rotationStepCount();
      int startIndex=index*n;
      int endIndex=startIndex+n;
      double sum=0.0;
      double ell=::tan(kell.z());
      double ell2=ell*ell;
      for(int i=startIndex; i<endIndex; i++) {
        FKPower p(&FKSteeringTwoComponentRayleighRadial::cache(_gridCache, i));
        p.setCaponValue(_crossSpectrum->rayleigh());
        sum+=p.value()*p.value()*ell2;
      }
      return sum;
    } else {
      return -1.0;
    }
  }

  double OmniFKRayleigh::value(const Point& kell) const
  {
    double k2=kell.x()*kell.x();
    if(k2>minimumK2() && k2<maximumK2()) {
      Point k;
      k.setZ(kell.y());
      int n=_crossSpectrum->rotationStepCount();
      double da=_crossSpectrum->rotationStep();
      double a;
      double sum=0.0;
      double ell=::tan(kell.z());
      double ell2=ell*ell;
      FKSteeringTwoComponentRayleighRadial * s=static_cast<FKSteeringTwoComponentRayleighRadial *>(_steering);
      FKPower p(s);
      for(int i=0; i<n; i++) {
        a=(double)i*da;
        k.setX(kell.x()*::cos(a));
        k.setY(kell.x()*::sin(a));
        s->initValue(k, ell);
        p.setCaponValue(_crossSpectrum->rayleigh());
        sum+=p.value()*p.value()*ell2;
      }
      return sum;
    } else {
      return -1.0;
    }
  }

} // namespace ArrayCore

