/***************************************************************************
**
**  This file is part of geopsy.
**
**  geopsy is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  geopsy is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-05-03
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <GeopsyGui.h>

#include "SignalTableItem.h"

SignalTableItem::SignalTableItem(const SubSignalPool * subPool, QObject * parent)
    : QAbstractItemModel(parent)
{
  TRACE;
  _subPool=subPool;
  _dataFields=GeopsyCoreEngine::instance()->preferences()->tableFields();
}

SignalTableItem::~SignalTableItem()
{
  TRACE;
}

void SignalTableItem::beginChange()
{
  TRACE;
  beginResetModel();
}

void SignalTableItem::endChange()
{
  TRACE;
  endResetModel();
}

int SignalTableItem::rowCount(const QModelIndex &parent) const
{
  TRACE;
  if( !parent.isValid())
    return _subPool->count();
  return 0;
}

int SignalTableItem::columnCount(const QModelIndex &) const
{
  TRACE;
  return _dataFields.count();
}

QVariant SignalTableItem::data(const QModelIndex &index, int role) const
{
  TRACE;
  if( !index.isValid()) return QVariant();
  Signal * sig=_subPool->at(index.row());
  switch (role) {
  case Qt::EditRole:
  case Qt::DisplayRole: {
      const MetaDataField& field=_dataFields.at(index.column());
      switch (field.id()) {
      case MetaDataFactory::Duration:
        return Number::secondsToDuration(sig->header(field).toDouble());
      default:
        if(field.id()==Comments::staticId()) {
          return sig->metaData<Comments>().value().trimmed();
        } else {
          return sig->header(field);
        }
      }
    }
  case Qt::ToolTipRole: {
      QString cmt=sig->metaData<Comments>().value();
      if(cmt.isEmpty()) {
        return QVariant();
      } else {
        return cmt.trimmed();
      }
    }
  default:
    return QVariant();
  }
}

bool SignalTableItem::setData(const QModelIndex & index, const QVariant & value, int role)
{
  TRACE;
  if( !index.isValid() ||
       index.row() >= _subPool->count() ||
       index.column() >=_dataFields.count()) return false;
  Signal * sig=_subPool->at(index.row());
  switch (role) {
  case Qt::EditRole: {
      const MetaDataField& field=_dataFields.at(index.column());
      GeopsyGuiEngine::instance()->beginSignalChange(sig);
      if(sig->setHeader(field, value) ) {
        sig->setHeaderModified(true);
      } else {
        SignalDatabase * db=database();
        if(db) {
          GeopsyCoreEngine::instance()->showMessage(db, tr("Modifying headers of signals: %1 is a read only field.")
                                  .arg(MetaDataFactory::instance()->name(field)));
        }
      }
      GeopsyGuiEngine::instance()->endSignalChange(sig);
    }
    return true;
  default:
    return false;
  }
}

QVariant SignalTableItem::headerData(int section, Qt::Orientation orientation, int role) const
{
  TRACE;
  if(role!=Qt::DisplayRole) return QVariant();
  if(orientation==Qt::Horizontal) {
    if(section<_dataFields.count())
      return _dataFields.at(section).title();
    else
      return QVariant();
  } else {
    return section + 1;
  }
}

QModelIndex SignalTableItem::parent(const QModelIndex &) const
{
  TRACE;
  return QModelIndex();
}

QModelIndex SignalTableItem::index(int row, int column, const QModelIndex &) const
{
  TRACE;
  if(column < 0 || column >=_dataFields.count() ||
       row < 0 || row >= _subPool->count()) return QModelIndex();
  return createIndex(row, column);
}
