/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-07-04
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "XMLSciFigs.h"

namespace SciFigs {

/*!
  \class XMLSciFigs XMLSciFigs.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

const QString XMLSciFigs::xmlSciFigsTag="SciFigs";

XMLSciFigs::XMLSciFigs()
{
  TRACE;
  _currentObject=nullptr;
  _currentFile=None;
  _layerMode=AppendLayer;
}

/*!
  Description still missing
*/
XMLClass::Error XMLSciFigs::saveFile(QString fileName, const XMLClass * object, FileTypes type)
{
  TRACE;
  if(type==None || !object) return XMLClass::NoError;
  _currentFile=type;
  _currentObject=object;
  XMLHeader hdr(this);
  XMLClass::Error err=hdr.xml_saveFile(fileName, this);
  _currentObject=nullptr;
  _currentFile=None;
  return err;
}

/*!
  Description still missing
*/
QString XMLSciFigs::saveString(const XMLClass * object, FileTypes type)
{
  TRACE;
  if(type==None || !object) return QString();
  _currentFile=type;
  _currentObject=object;
  XMLHeader hdr(this);
  QString res=hdr.xml_saveString(true, this);
  _currentObject=nullptr;
  _currentFile=None;
  return res;
}

/*!
  Description still missing
*/
QByteArray XMLSciFigs::saveByteArray(const XMLClass * object, FileTypes type)
{
  TRACE;
  if(type==None || !object) return QByteArray();
  _currentFile=type;
  _currentObject=object;
  XMLHeader hdr(this);
  QByteArray res=hdr.xml_saveByteArray(this);
  _currentObject=nullptr;
  _currentFile=None;
  return res;
}

/*!
  Description still missing
*/
XMLClass::Error XMLSciFigs::restoreFile(QString fileName, XMLClass * object, FileTypes type)
{
  TRACE;
  if(type==None || !object) return XMLClass::NoError;
  _currentFile=type;
  _currentObject=object;
  XMLHeader hdr(this);
  hdr.addAlternateTag("Scifigs"); // compatibility
  XMLClass::Error err=hdr.xml_restoreFile(fileName, this);
  _currentObject=nullptr;
  _currentFile=None;
  return err;
}

/*!
  Description still missing
*/
XMLClass::Error XMLSciFigs::restoreString(QString s, XMLClass * object, FileTypes type)
{
  TRACE;
  if(type==None || !object) return XMLClass::NoError;
  _currentFile=type;
  _currentObject=object;
  XMLHeader hdr(this);
  hdr.addAlternateTag("Scifigs"); // compatibility
  XMLClass::Error err=hdr.xml_restoreString(s, this);
  _currentObject=nullptr;
  _currentFile=None;
  return err;
}

/*!
  Description still missing
*/
XMLClass::Error XMLSciFigs::restoreByteArray(QByteArray data, XMLClass * object, FileTypes type)
{
  TRACE;
  if(type==None || !object) return XMLClass::NoError;
  _currentFile=type;
  _currentObject=object;
  XMLHeader hdr(this);
  hdr.addAlternateTag("Scifigs"); // compatibility
  XMLClass::Error err=hdr.xml_restoreByteArray(data, this);
  _currentObject=nullptr;
  _currentFile=None;
  return err;
}

/*!
  Description still missing
*/
void XMLSciFigs::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
{
  TRACE;
  Q_UNUSED(context)
  writeProperty(s, "libVersion", PackageInfo::package("SciFigs")->version());
  switch (_currentFile & Page) {
  case None:
    writeProperty(s, "type", QString("None"));
    break;
  case MakeUp:
    writeProperty(s, "type", QString("MakeUp"));
    break;
  case Data:
    writeProperty(s, "type", QString("Data"));
    break;
  case Layer:
    writeProperty(s, "type", QString("Layer"));
    break;
  case Page:
    writeProperty(s, "type", QString("Page"));
    break;
  }
}

/*!
  Description still missing
*/
void XMLSciFigs::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
{
  TRACE;
  /* The name is not put in the attribute as there is always only one
     child in this context */
  if(_currentObject) {
    if((_currentFile & PageHeader) && _currentObject->xml_tagName()!="GraphicSheet") {
      s << s.indent() << "<GraphicSheet>\n";
      s.incrementIndent();
      static const QString key("objectName");
      XMLSaveAttributes att;
      att.add(key);
      _currentObject->xml_save(s, context, att);
      s.decrementIndent();
      s << s.indent() << "</GraphicSheet>" << endl;
    } else {
      _currentObject->xml_save(s, context);
    }
  }
}

XMLMember XMLSciFigs::xml_member(XML_MEMBER_ARGS)
{
  TRACE;
  Q_UNUSED(attributes)
  Q_UNUSED(context)
  if(tag=="xmlVersion") return XMLMember(0);
  else if(tag=="libVersion") return XMLMember(1);
  else if(tag=="type") return XMLMember(2);
  else if(_currentObject) {
    if(tag==_currentObject->xml_tagName() ||
       _currentObject->xml_isValidTagName(tag.toString())) {
      return XMLMember(const_cast<XMLClass *>(_currentObject));
    } else {
      Message::warning(MSG_ID, tr("Parsing Scifig XML"), tr("The XML information is for object of type %1. "
                           "The expected type is %2.").arg(tag.toString()). arg(_currentObject->xml_tagName()),
                           true);
      return XMLMember(XMLMember::Unknown);
    }
  } else {
    return XMLMember(XMLMember::Unknown);
  }
}

bool XMLSciFigs::xml_setProperty(XML_SETPROPERTY_ARGS)
{
  TRACE;
  Q_UNUSED(tag)
  Q_UNUSED(attributes)
  Q_UNUSED(context)
  switch (memberID) {
  case 0:
    // left here to avoid error when reading old SciFigs files
    return true;
  case 1: {
      Version v(content.toString());
      static const Version vRef(PackageInfo::package("SciFigs")->version());
      if(v>vRef) {
        App::log(tr("This file was generated by SciFigs %1, which is more recent\n"
                    "than your current version %2.\n"
                    "Upgrading might be necessary.\n").arg(content.toString()).arg(vRef.toString()));
      }
      _currentFileVersion=content.toString();
      return true;
    }
  case 2: {
      FileType fileType=None;
      if(content=="MakeUp") fileType=MakeUp;
      else if(content=="Data") fileType=Data;
      else if(content=="Layer") fileType=Layer;
      else if(content=="Page") fileType=Page;
      if(fileType!=_currentFile) {
        if(fileType==Data && _currentFile==Layer) { // Compatibility: before 20110106, layer is saved as data
          return true;
        }
        App::log(tr("No information to restore from file type %1\n").arg(content.toString()) );
        return false;
      }
    }
    return true;
  default:
    return false;
  }
}

} // namespace SciFigs
