/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-03-23
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "SciFigsPreferences.h"
#include "SciFigsPlugin.h"
#include "SciFigsGlobal.h"
#include "PluginSettings.h"

namespace SciFigs {

SciFigsPreferences::SciFigsPreferences(QWidget* parent, Qt::WindowFlags f) :
    Dialog(parent, f)
{
  TRACE;
  setupUi(this);

  SciFigsPluginSettings reg;
  pathList->addItems(reg.paths());
  setPluginList(reg.pluginList());
}

SciFigsPreferences::~SciFigsPreferences()
{
  TRACE;
}

void SciFigsPreferences::savePluginList()
{
  TRACE;
  QStringList paths=getPaths();
  SciFigsPluginSettings reg;
  reg.setPaths(paths);
  QStringList tools=getPluginList();
  reg.setPluginList(tools);
}

void SciFigsPreferences::on_searchPlugin_clicked()
{
  TRACE;
  QStringList paths=getPaths();
  pluginList->clear();
  bool ok;
  QStringList plugins=SciFigsPlugin::getPluginList(File::getLibList(paths), ok);
  if(!ok) {
    Message::warning(MSG_ID, tr("Searching SciFigs plugins"), tr("One ore more plugins in your search path(s) "
                          "are not compatible with this SciFigs version. It is likely that old versions of plugins "
                          "were not de-installed properly."));
  }
  setPluginList(plugins);
}

void SciFigsPreferences::on_addPath_clicked()
{
  TRACE;
  QString path=Message::getExistingDirectory(tr("Add plugin search path"));
  if(!path.isEmpty()) pathList->addItem(path);
}

void SciFigsPreferences::on_removePath_clicked()
{
  TRACE;
  delete pathList->takeItem(pathList->currentRow());
}

QStringList SciFigsPreferences::getPaths()
{
  TRACE;
  QStringList paths;
  int n=pathList->count();
  for(int i=0;i<n;i++) {
    paths << pathList->item(i)->text();
  }
  return paths;
}

QStringList SciFigsPreferences::getPluginList()
{
  TRACE;
  QStringList pluginFiles;
  int n=pluginList->count();
  for(int i=0;i<n;i++) {
    pluginFiles << pluginList->item(i)->data(32).toString();
  }
  return pluginFiles;
}

void SciFigsPreferences::setPluginList(QStringList plugins)
{
  TRACE;
  int n=plugins.count();

  for(int i=0;i<n;i++) {
    QString libFile=plugins.at(i);
    if(!libFile.isEmpty()) {
      SciFigsPlugin * pluginInstance=SciFigsPlugin::loadPlugin(libFile);
      if(pluginInstance) {
        if(pluginInstance->isValid()) {
          QListWidgetItem * item=new QListWidgetItem(pluginInstance->title());
          item->setWhatsThis(pluginInstance->description());
          item->setData(32,libFile);
          pluginList->addItem(item);
        }
        CoreApplication::instance()->deletePlugin(pluginInstance);
      }
    }
  }
}

} // namespace SciFigs
