/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2002-05-24
**  Copyright: 2002-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef LEGEND_H
#define LEGEND_H

#include <QGpCoreMath.h>

#include "LegendItem.h"
#include "SciFigsDLLExport.h"

namespace SciFigs {

class Pen;
class Symbol;

class SCIFIGS_EXPORT Legend : public XMLClass, public AbstractColorPalette
{
public:
  Legend() {_textLookup=nullptr;}
  Legend(const Legend& o) : XMLClass() {_textLookup=nullptr; operator=(o);}
  Legend(int n) {_textLookup=nullptr; resize(n);}
  virtual ~Legend();

  Legend& operator=(const Legend& o);
  bool operator==(const Legend& o) const;
  bool operator!=(const Legend& o) const {return !operator==(o);}
  bool operator<(const Legend& o) const;

  virtual const QString& xml_tagName() const {return xmlLegendTag;}
  static const QString xmlLegendTag;

  void resize(int n);
  void clear();
  void setTexts(const QStringList& texts, const Pen& defaultPen, const Symbol& defaultSymbol);

  virtual ColorPalette palette() const;
  virtual void setPalette(const ColorPalette& pal);
  virtual void generateColorScale(int n, ColorPalette::Model m=ColorPalette::McNamesClip,
                                  bool reversed=false, int transparency=255);
  virtual void generateGrayScale(int n, ColorPalette::Model m=ColorPalette::McNamesClip,
                                 bool reversed=false, int transparency=255);

  const LegendItem& item(int i);

  virtual int count() const {return _items.size();}
  int indexOf(const QString& text) const;

  void setText(int i, QString t);
  const QString& text(int i) const;

  void setTextColor(int i, QColor c);
  const QColor& textColor(int i) const;

  void setPen(int i, Pen p);
  const Pen& pen(int i) const;
  const Pen& pen(const QString& text) const;

  void setSymbol(int i, Symbol s);
  const Symbol& symbol(int i) const;
  const Symbol& symbol(const QString& text) const;

  QStringList texts() const;
  void setTexts(const Legend& o);
  void setTextColors(const Legend& o);
  void setPens(const Legend& o);
  void setSymbols(const Legend& o);

  void penColorToSymbolPen();
  void penColorToSymbolBrush();
  void penColorToText();
protected:
  virtual void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
private:
  void resetTextLookup();

  QVector<LegendItem> _items;
  mutable QMap<QString, int> * _textLookup;
};

//QTextStream& operator>>(QTextStream& s, Legend& p);

} // namespace SciFigs

// Allow passing Legend as a QVariant
Q_DECLARE_METATYPE(SciFigs::Legend)

#endif // LEGEND_H
