/***************************************************************************
**
**  This file is part of HVCore.
**
**  HVCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  HVCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-08-17
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef CURVERESULTS_H
#define CURVERESULTS_H

#include "AbstractResults.h"

#include "HVCoreDLLExport.h"

namespace HVCore {

  class HVCORE_EXPORT CurveResults : public AbstractResults
  {
    Q_OBJECT
  public:
    CurveResults(AbstractStation * parent);
    ~CurveResults();

    enum PeakValue {PeakFrequency, PeakAmplitude};

    void clear();
    void setWindows(int winCount, const HVParameters& param);
    void addWindows(int winCount);
    int windowCount() const {return _windows->curveCount();}
    void setWindow(int winIndex, double winLength, const Curve<Point2D>& curve);
    void setWindowColor(int winIndex, const Color& col);
    void setWindowColor(const QVector<int>* selCurves, const Color& col);
    void setStatistics();
    void setPeaks();
    QString comments() const;
    bool save(QString fileName, QString log) const;
    bool load(QString fileName) {return loadAverage(fileName);}

    void clearPeaks();
    void addPeak(double minFreq, double maxFreq);
    void setPeak(int index, double minFreq, double maxFreq);
    void removePeak(int index);

    void blackWindows(const Rect &rect);

    int peakCount() const {return _peaks.count();}
    int windowPeakCount(int index) const {return _peaks.at(index).weight();}
    double peak(int index, PeakValue pv) const;
    double peakMinimum(int index) const;
    double peakMaximum(int index) const;
    double peak(double min, double max, PeakValue pv) const;
    double lowestPeak(PeakValue pv) const;
    double highestPeak(PeakValue pv) const;
    double maximumAmplitudePeak(PeakValue pv) const;
    double maximumAmplitude() const;
    double averageWindowLength() const {return _averageWindowLength.mean();}

    XUniqueYColorData * windows() const {return _windows;}
    const Curve<Point2D>& average() const {return _average;}
    const Curve<Point2D>& stddevLow() const {return _stddevLow;}
    const Curve<Point2D>& stddevHigh() const {return _stddevHigh;}

    void printAverage() const;
    void printWindows() const;
    void printPeaks() const;

    void derivate();
    void integrate();
  private slots:
    void blackWindows(int, const QVector<int>* selCurves);
  signals:
    void colorsToChange(const QVector<int>* selCurves, const Color& col);
    void showColors(bool);
    void peakChanged();
  private:
    void createAverageLines();
    void setPeak(int index, int minFreq, int maxFreq);
    int averageMax(const Curve<Point2D>& meanCurve) const;
    int max(const double * y, int minFreq, int maxFreq) const;
    bool saveLog(QString fileName, QString log) const;
    bool saveAverage(QString fileName) const;
    bool loadAverage(QString fileName);

    XUniqueYColorData * _windows;
    Curve<Point2D> _average, _stddevLow, _stddevHigh;
    QVector<RealStatisticalValue> _peaks;
    Statistics _averageWindowLength;
  };

} // namespace HVCore

#endif // CURVERESULTS_H
