/***************************************************************************
**
**  This file is part of DinverDCGui.
**
**  DinverDCGui is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverDCGui is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-01-18
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <DinverDCCore.h>
#include <QGpGuiTools.h>
#include <QGpGuiWave.h>
#include <SciFigs.h>
#include "DispersionViewer.h"

namespace DinverDCGui {

DispersionViewer::DispersionViewer(QWidget * parent)
    : DCModelViewer(parent)
{
  TRACE;
  setObjectName("DispersionViewer");

  // Tools menu
  QMenu * m;
  QAction * a;

  m=addMenu(tr("&Tools"));

  a=new QAction(tr("Current target"), this);
  a->setCheckable(true);
  a->setChecked(true);
  a->setStatusTip(tr("Show/hide the current target onto the inverted dispersion curves"));
  connect(a, SIGNAL(toggled(bool)), this, SLOT(showTarget(bool)));
  m->addAction(a);
}

DispersionViewer::~DispersionViewer()
{
  TRACE;
}

void DispersionViewer::setMode(const Mode& m)
{
  TRACE;
  _mode=m;
  setWindowTitle(tr("Dispersion curves (%1 %2):")
                 .arg(Mode::userSlowness(m.slowness()))
                 .arg(Mode::userPolarization(m.polarization())));
}

AbstractLine * DispersionViewer::targetReferenceLine() const
{
  TRACE;
  ModalLine * line=new ModalLine;
  line->setPen(Pen(Qt::black, 0.6));
  line->setSymbol(Symbol( Symbol::Circle, 1.2, Pen(Qt::black, 0.0),
                              Brush(Qt::black, Qt::SolidPattern)));
  return line;
}

void DispersionViewer::setTitles(int i)
{
  TRACE;
  AxisWindow * w=_graphs[i];
  w->yAxis()->setTitle("Slowness (s/m)");
  w->yAxis()->setTitleInversedScale("Velocity (m/s)");
  w->yAxis()->setScaleType(Scale::InversedLog);
  w->xAxis()->setTitle("Frequency (Hz)");
  w->xAxis()->setTitleInversedScale("Period (s)");
  w->xAxis()->setScaleType(Scale::Log);
  w->xAxis()->setAutoTicks(false);
  w->xAxis()->setMajorTicks(2.0);
  w->xAxis()->setMinorTicks(0.5);
  w->setObjectName(QString("dc_%1").arg(i));
}

bool DispersionViewer::hasCurves(DCReportBlock& dcBlock, int)
{
  TRACE;
  if(dcBlock.dispersion()) {
    dcBlock.readModalStorages(4);
    return seekMode(dcBlock);
  } else return false;
}

void DispersionViewer::setSampleCount(DCModelInfo * info, DCReportBlock& dcBlock)
{
  TRACE;
  ModalStorageReader reader(dcBlock.stream());
  int n=reader.nModes();
  if(n>_nGraphs) n=_nGraphs;
  int i, nf;
  for(i=0;i<n;i++) {
    if(reader.seek(dcBlock.stream(),i)) {
      dcBlock.stream() >> nf;
      info->setSampleCount(i,nf);
    } else {
      info->setSampleCount(i,0);
    }
  }
  for(;i<_nGraphs;i++) info->setSampleCount(i,0);
}

void DispersionViewer::report2plot(DCReportBlock& dcBlock, Point2D ** points, const DCModelInfo& info)
{
  TRACE;
  ModalStorageReader reader(dcBlock.stream());
  int n=reader.nModes();
  if(n>_nGraphs) n=_nGraphs;
  for(int i=0;i<n;i++) {
    reader.toPlot(dcBlock.stream(), points[i]+info.offset(i));
  }
}

void DispersionViewer::setLimits(int i, const Rect& r)
{
  TRACE;
  _graphs[ i ] ->xAxis()->setRange(r.x1() * 0.95, r.x2() * 1.05);
  _graphs[ i ] ->yAxis()->setRange(r.y1() * 0.95, r.y2() * 1.05);
}

bool DispersionViewer::seekMode(DCReportBlock& dcBlock)
{
  TRACE;
  switch (_mode.polarization()) {
  case Mode::Love:
    switch (_mode.slowness()) {
    case Mode::Phase:
      return dcBlock.modalStorage(2);
    case Mode::Group:
      return dcBlock.modalStorage(3);
    }
    break;
  default:
    switch (_mode.slowness()) {
    case Mode::Phase:
      return dcBlock.modalStorage(0);
    case Mode::Group:
      return dcBlock.modalStorage(1);
    }
    break;
  }
  return false;
}

void DispersionViewer::setTarget(const TargetList * tl)
{
  TRACE;
  const QList<ModalCurve>& curves=tl->dispersionTarget().curves();
  for(int i=0;i<_nGraphs;i++) {
    _targetLayers[i]->setErrorBar(LineLayer::VerticalLogBar);
    _mode.setIndex(i);
    for(QList<ModalCurve>::const_iterator it=curves.begin();it!=curves.end(); it++) {
      if(it->hasMode(_mode) && it->isEnabled()) {
        static_cast<ModalLine *>(_targetLayers[i]->addLine())->setCurve(*it);
      }
    }
    if(i==0)
      addText(i, tr("Fundamental mode"));
    else
      addText(i, tr("Higher mode %1").arg(i));
  }
}

} // namespace DinverDCGui
