/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2017-06-08
**  Copyright: 2017-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef SUMCONDITION_H
#define SUMCONDITION_H

#include "AbstractCondition.h"
#include "DinverCoreDLLExport.h"

namespace DinverCore {

  class DINVERCORE_EXPORT SumCondition : public AbstractCondition
  {
  public:
    SumCondition(Type type, double constant=1.0);
    ~SumCondition();
    virtual bool operator==(const AbstractCondition& o) const;

    virtual const QString& xml_tagName() const {return xmlSumConditionTag;}
    static const QString xmlSumConditionTag;

    void addParameter(Parameter * p, double factor=1.0);

    void setFactor(int index, double value);

    inline virtual void getLimits(int paramIndex, double& min, double& max) const;
    virtual bool adjustRanges();
    virtual void humanInfo() const;
  protected:
    virtual uint internalChecksum() const;
    virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
    virtual XMLMember xml_member(XML_MEMBER_ARGS);
    virtual bool xml_setProperty(XML_SETPROPERTY_ARGS);
  private:
    Type _type;
    double * _factors;
    double _constant;
  };

} // namespace DinverCore


#include "Parameter.h"

namespace DinverCore {

  inline void SumCondition::getLimits(int paramIndex, double& min, double& max) const
  {
    TRACE;
    double threshold=_constant;
    int i;
    if(_factors) {
      for(i=count()-1; i>paramIndex; i--) {
        threshold-=_factors[i]*parameter(i).realValue();
      }
      // Skip paramIndex
      for(i--; i>=0; i--) {
        threshold-=_factors[i]*parameter(i).realValue();
      }
      switch (_type) {
      case LessThan:
        if(threshold>_factors[paramIndex]*min) {
          min=threshold/_factors[paramIndex];
        }
        break;
      case GreaterThan:
        if(threshold<_factors[paramIndex]*max) {
            max=threshold/_factors[paramIndex];
        }
        break;
      case NoCondition:
        break;
      }
    } else {
      for(i=count()-1; i>paramIndex; i--) {
        threshold-=parameter(i).realValue();
      }
      // Skip paramIndex
      for(i--; i>=0; i--) {
        threshold-=parameter(i).realValue();
      }
      switch (_type) {
      case LessThan:
        if(threshold<max) {
          max=threshold;
        }
        break;
      case GreaterThan:
        if(threshold>min) {
          min=threshold;
        }
        break;
      case NoCondition:
        break;
      }
    }
  }

} // namespace DinverCore

#endif // SUMCONDITION_H

