/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-02-26
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "SimpleCondition.h"
#include "Parameter.h"

namespace DinverCore {

/*!
  \class SimpleCondition SimpleCondition.h
  \brief A simple condition p1 < or > a*p2+b

*/

const QString SimpleCondition::xmlSimpleConditionTag="SimpleCondition";

/*!
  Define a condition between \a p1 and \a p2.

  Order of arguments should be read as:

  \a p1 LessThan or GreaterThan (\a type) \a factor times \a p2 plus \a constant
*/
SimpleCondition::SimpleCondition(Parameter * p1, Type type, double factor, Parameter * p2, double constant)
{
  TRACE;
  addParameter(p1);
  addParameter(p2);
  _type=type;
  _factor=factor;
  _constant=constant;
}

bool SimpleCondition::operator==(const AbstractCondition& o) const
{
  TRACE;
  if(xml_tagName()==o.xml_tagName()) {
    const SimpleCondition& sc=static_cast<const SimpleCondition&>(o);
    return _type==sc._type &&
          _factor==sc._factor &&
          _constant==sc._constant &&
          AbstractCondition::operator==(o);
  } else return false;
}

/*!
  \fn void SimpleCondition::getLimits(int paramIndex, double& min, double& max) const
  Adjust \a min or/and \a max to fulfill the condition. \a paramIndex must be 0 or 1.
*/

/*!
  Return true if original ranges of parameter are touched. Ranges can be reduced, never enlarged
*/
bool SimpleCondition::adjustRanges()
{
  TRACE;
  double min, max;
  bool modified=false;
  Parameter& p0=parameter(0);
  Parameter& p1=parameter(1);
  switch (_type) {
  case LessThan: {
      p0.setRealValue(p0.minimum());
      min=-std::numeric_limits<double>::infinity();
      getLimits(1, min, max);
      if(p1.minimum()<min) {
        p1.setMinimum(min);
        modified=true;
      }
      p1.setRealValue(p1.maximum());
      max=std::numeric_limits<double>::infinity();
      getLimits(0, min, max);
      if(p0.maximum()>max) {
        p0.setMaximum(max);
        modified=true;
      }
    }
    break;
  case GreaterThan: {
      p0.setRealValue(p0.maximum());
      max=std::numeric_limits<double>::infinity();
      getLimits(1, min, max);
      if(p1.maximum()>max) {
        p1.setMaximum(max);
        modified=true;
      }
      p1.setRealValue(p1.minimum());
      min=-std::numeric_limits<double>::infinity();
      getLimits(0, min, max);
      if(p0.minimum()<min) {
        p0.setMinimum(min);
        modified=true;
      }
    }
    break;
  case NoCondition:
    break;
  }
  return modified;
}

/*!
  Return a number that uniquely identify this condition
*/
uint SimpleCondition::internalChecksum() const
{
  TRACE;
  uint cs=0;
  cs+=qHash(static_cast<qint64>(_factor));
  cs+=qHash(static_cast<qint64>(_constant));
  cs+=_type+1;
  return cs;
}

/*!
  Print human readable information about condition to stream.
*/
void SimpleCondition::humanInfo() const
{
  TRACE;
  App::log("        "+parameter(0).name()+" "+(_type==LessThan ? "<" : ">" )+" ");
  if(_factor!=1.0)
    App::log(QString::number(_factor)+"*");
  App::log(parameter(1).name());
  if(_constant>0) {
    App::log("+"+QString::number(_constant));
  } else if(_constant<0) {
    App::log(QString::number(_constant));
  }
  App::log("\n");
}

void SimpleCondition::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
{
  TRACE;
  Q_UNUSED(context)
  writeProperty(s, "type", toString(_type));
  writeProperty(s, "factor", _factor);
  writeProperty(s, "constant", _constant);
}

XMLMember SimpleCondition::xml_member(XML_MEMBER_ARGS)
{
  TRACE;
  if(tag=="factor")
    return XMLMember(0);
  else if(tag=="constant")
    return XMLMember(1);
  else if(tag=="type" )
    return XMLMember(2);
  else
    return SimpleCondition::xml_member(tag, attributes, context)+3;
}

bool SimpleCondition::xml_setProperty(XML_SETPROPERTY_ARGS)
{
  TRACE;
  switch (memberID) {
  case 0:
    _factor=content.toDouble();
    return true;
  case 1:
    _constant=content.toDouble();
    return true;
  case 2: {
      bool ok=true;
      _type=fromString(content.toString(), ok);
      return ok;
    }
  default:
    return AbstractCondition::xml_setProperty(memberID-3, tag, attributes, content, context);
  }
}

} // namespace DinverCore
