/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-17
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef PARAMETERGRID_H
#define PARAMETERGRID_H

#include <QGpCoreMath.h>

#include "DinverCoreDLLExport.h"

namespace DinverCore {

class DINVERCORE_EXPORT ParameterGrid
{
public:
  ParameterGrid();
  ~ParameterGrid();

  enum Scale {Linear, Log};

  void setScale(Scale s) {_scale=s;}
  void setPrecision(double p) {ASSERT(p>=0.0); _precision=p;}

  void setMinimum(double m) {_minimum=m;}
  double minimum() const {return _minimum;}

  void setMaximum(double m) {_maximum=m;}
  double maximum() const {return _maximum;}

  bool init();

  int count() const {return _n;}
  inline double realValue(int index) const;
  inline double realValue(double index) const;
  inline int gridValue(double val) const;
  inline void getLimits(int& min, int& max, double dmin, double dmax) const;
  inline void getLimits(double& min, double& max, double dmin, double dmax) const;
private:
  Scale _scale;
  double _minimum;
  double _maximum;
  double _precision;
  double * _realValues;
  int _n;
  double _a, _b, _c;
};

inline double ParameterGrid::realValue(int index) const
{
  return _realValues[index];
}

inline double ParameterGrid::realValue(double index) const
{
  switch(_scale) {
  case Linear:
    break;
  case Log:
    return _minimum*pow(_c, index);
  }
  return _minimum+index*_b;
}

inline int ParameterGrid::gridValue(double val) const
{
  switch(_scale) {
  case Linear:
    break;
  case Log:
    return qRound(log(val*_a) * _b);
  }
  return qRound((val-_minimum)*_a);
}

inline void ParameterGrid::getLimits(int& min, int& max, double dmin, double dmax) const
{
  switch (_scale) {
  case Linear:
    if(dmin==_minimum) {
      min=0;
    } else {
      min=qCeil((dmin-_minimum)*_a+1e-10);
      if(min>_n) {
        min=_n;
      }
    }
    if(dmax==_maximum) {
      max=_n;
    } else {
      max=qFloor((dmax-_minimum)*_a-1e-10);
      if(max<0) {
        max=0;
      }
    }
    break;
  case Log:
    if(dmin==_minimum) {
      min=0;
    } else {
      min=qCeil(log(dmin*_a)*_b+1e-10);
      if(min>_n) {
        min=_n;
      }
    }
    if(dmax==_maximum) {
      max=_n;
    } else {
      max=qFloor(log(dmax*_a)*_b-1e-10);
      if(max<0) {
        max=0;
      }
    }
    break;
  }
}

inline void ParameterGrid::getLimits(double& min, double& max, double dmin, double dmax) const
{
  switch (_scale) {
  case Linear:
    if(dmin==_minimum) {
      min=0;
    } else {
      min=(dmin-_minimum)*_a;
    }
    if(dmax==_maximum) {
      max=_n;
    } else {
      max=(dmax-_minimum)*_a;
    }
    break;
  case Log:
    if(dmin==_minimum) {
      min=0;
    } else {
      min=log(dmin* _a)*_b;
    }
    if(dmax==_maximum) {
      max=_n;
    } else {
      max=log(dmax*_a)*_b;
    }
    break;
  }
}

} // namespace DinverCore

#endif // PARAMETERGRID_H
