/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-02-02
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKRayleighEllipticity.h"
#include "FKGridSearch.h"
#include "FKCrossSpectrum.h"
#include "FKPower.h"
#include "FKSteeringTwoComponentRayleighRadial.h"
#include "FKSteeringTwoComponentRayleighVertical.h"
#include "FKSteeringThreeComponentRayleighRadial.h"
#include "FKSteeringThreeComponentRayleighVertical.h"

namespace ArrayCore {

  /*!
    \class HRFKRayleighEllipticity HRFKRayleighEllipticity.h
    \brief Rayleigh high resolution FK power validator

    Base class to get Rayleigh FK power with radial and vertical steering.
    Both are required to compute the noise estimation as in Wathelet et al (2018) in GJI.

    Cross-spectrum must be set because the direction of projection is not known by this object.
    This is mandatory for RTBF processing. For direct steering this object directly accesses process
    to get the cross-spectrum.
  */

  HRFKRayleighEllipticity::HRFKRayleighEllipticity(FKCache * cache)
    : HRFKRayleigh(cache)
  {
    delete _steering;
    _steering=new FKSteeringTwoComponentRayleighRadial;
    _steering->setArray(cache->array());
    static_cast<FKSteeringTwoComponentRayleighRadial *>(_steering)->resize();

    _crossSpectrum=nullptr;
  }

  double HRFKRayleighRadialEllipticity::value(const Point& ell) const
  {
    FKSteeringTwoComponentRayleighRadial * s=static_cast<FKSteeringTwoComponentRayleighRadial *>(_steering);
    s->initValue(_k, ::tan(ell.x()));
    FKPower p(s);
    p.setCaponValue(*_crossSpectrum);
    return p.value();
  }

  double HRFKRayleighVerticalEllipticity::value(const Point& ell) const
  {
    FKSteeringTwoComponentRayleighVertical * s=static_cast<FKSteeringTwoComponentRayleighVertical *>(_steering);
    s->initValue(_k, ::tan(ell.x()));
    FKPower p(s);
    p.setCaponValue(*_crossSpectrum);
    return p.value();
  }

  HRFKDirectRayleighEllipticity::HRFKDirectRayleighEllipticity(FKCache * cache)
    : HRFKRayleighEllipticity(cache)
  {
    delete _steering;
    _steering=new FKSteeringThreeComponentRayleighRadial;
    _steering->setArray(cache->array());
    static_cast<FKSteeringThreeComponentRayleighRadial *>(_steering)->resize();
  }

  double HRFKDirectRayleighRadialEllipticity::value(const Point& ell) const
  {
    FKSteeringThreeComponentRayleighRadial * s=static_cast<FKSteeringThreeComponentRayleighRadial *>(_steering);
    s->initValue(_k, ::tan(ell.x()));
    FKPower p(s);
    p.setCaponValue(*_crossSpectrum); // Doubt if it is the right crossspectrum, check it
    return p.value();
  }

  double HRFKDirectRayleighVerticalEllipticity::value(const Point& ell) const
  {
    FKSteeringThreeComponentRayleighVertical * s=static_cast<FKSteeringThreeComponentRayleighVertical *>(_steering);
    s->initValue(_k, ::tan(ell.x()));
    FKPower p(s);
    p.setCaponValue(*_crossSpectrum); // Doubt if it is the right crossspectrum, check it
    return p.value();
  }

} // namespace ArrayCore

