/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-02-02
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ConvFKRayleigh.h"
#include "FKCrossSpectrum.h"
#include "FKPower.h"
#include "FKGridSearch.h"
#include "FKSteeringTwoComponentRayleighRadial.h"

namespace ArrayCore {

  /*!
    \class ConvFKRayleigh ConvFKRayleigh.h
    \brief Rayleigh conventional FK power

    Rayleigh conventional FK power versus wavenumber (k) function.
    The cross-spectrum is already projected on the radial direction.
  */

  ConvFKRayleigh::ConvFKRayleigh(FKCache * gridCache)
    : AbstractFKFunction(gridCache)
  {
    delete _steering;
    _steering=new FKSteeringTwoComponentRayleighRadial;
    _steering->setArray(gridCache->array());
    static_cast<FKSteeringTwoComponentRayleighRadial *>(_steering)->resize();
  }

  void ConvFKRayleigh::resetCrossSpectrum()
  {
    _crossSpectrum->resetRotatedRayleigh();
  }

  void ConvFKRayleigh::addCrossSpectrum()
  {
    _crossSpectrum->addRotatedRayleigh();
  }

  void ConvFKRayleigh::meanCrossSpectrum(int nBlocks)
  {
    _crossSpectrum->meanRotatedRayleigh(nBlocks);
  }

  bool ConvFKRayleigh::initGrid(int n)
  {
    if(_gridCache->isEmpty(FKCache::TwoComponentRayleigh)) {
      App::log(tr("Caching steering vectors for 2-component Rayleigh ... (%1 values)\n").arg(n) );
      _gridCache->resize(FKCache::TwoComponentRayleigh, n);
      return true;
    } else {
      return false;
    }
  }

  void ConvFKRayleigh::setGrid(FKGridSearch * g, double step, double size)
  {
    // Binary prograde/retrograde inversion
    g->setGrid(-size, size, step, -size, size, step, -0.25*M_PI, 0.25*M_PI, 0.5*M_PI);
    // Rayleigh prograde
    //g->setGrid(-size, size, step, -size, size, step, 0.25*M_PI, 0.25*M_PI, 0.0);
    // Rayleigh retrograde
    //g->setGrid(-size, size, step, -size, size, step, -0.25*M_PI, -0.25*M_PI, 0.0);
  }

  void ConvFKRayleigh::initGrid(const Point& kell, int index)
  {
    FKSteeringTwoComponentRayleighRadial::init(_gridCache, index , kell);
  }

  double ConvFKRayleigh::value(const Point& kell, int index) const
  {
    if(isInsideLimits(kell)) {
      FKPower p(&FKSteeringTwoComponentRayleighRadial::cache(_gridCache, index));
      p.setConventionalValue(_crossSpectrum->rotatedRayleigh(kell));
      return p.value();
    } else {
      return -1.0;
    }
  }

  double ConvFKRayleigh::value(const Point& kell) const
  {
    if(isInsideLimits(kell)) {
      FKSteeringTwoComponentRayleighRadial * s=static_cast<FKSteeringTwoComponentRayleighRadial *>(_steering);
      double ell=::tan(kell.z());
      s->initValue(kell, ell);
      FKPower p(s);
      p.setConventionalValue(_crossSpectrum->rotatedRayleigh(kell));
      return p.value();
    } else {
      return -1.0;
    }
  }

} // namespace ArrayCore

