/***************************************************************************
**
**  This file is part of geopsy.
**
**  geopsy is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  geopsy is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-12-01
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef MainWindow_H
#define MainWindow_H

#include <GeopsyGui.h>
#include <GeopsySLinkGui.h>
#include <QGpGuiTools.h>

class ToolFactory;
class FileView;
class GroupWidget;
class SeismicEventWidget;
class WaveformConsole;

class MainWindow : public MultiDocumentWindow
{
  Q_OBJECT
public:
  MainWindow(QWidget *parent=nullptr, Qt::WindowFlags f=Qt::Widget);
  ~MainWindow();

  QLabel * databaseInfo;

  SignalDatabase * database() const {return _db;}

  void initTools();
  SubPoolWindow * currentSubPoolWin() {return _subPoolWin;}

  ToolFactory * toolFactory() {return _toolFactory;}

  ProgressBar * progress() const {return _progressBar;}
  const QAction * viewTableAction() const {return _viewTableAction;}
  const QAction * viewGraphicAction() const {return _viewGraphicAction;}
  const QAction * viewMapAction() const {return _viewMapAction;}
  const QAction * viewChronogramAction() const {return _viewChronogramAction;}

  bool isTableEditionLocked() {return _tableEditionLocked;}
  virtual bool useProgress() {return true;}
  virtual QStatusBar * statusBar() {return QMainWindow::statusBar();}
  void startupPreferences();
  static QAction * newAction(const QAction& original, QWidget * parent=nullptr);
  void filesRemoved(const QList<SignalFile *>& fList);

  void beginAddFile();
  void endAddFile();
public slots:
  void activeWindowChanged(MultiDocumentSubWindow * w);
  void setSubPoolActionEnabled(bool b, bool rw);

  void clearAll();
  void exportSignal();
  void importTable();
  void setPreferences();
  void saveDBAs();
  void openDB(QString fileName=QString());
  virtual void openDocument(const QString& fileName) {openDB(fileName);}
  void saveDB();
  void load();
  void loadFilePattern();
  void directoryMonitor();
  void seedlink(QByteArray serverAddress=nullptr, qint16 serverPort=0,
                QString streamSelection=QString());
  void loadCity();
  void quit(bool force=false);

  // Edit menu
  void setTableLocked(bool l);
  void sort();
  void setHeader();
  void setReceivers();
  void createNewGroup();
  // View menu
  void viewSetDataFields();
  void fftTime();
  void fftAmpl();
  void fftPhase();
  void showToolBarMenu();

  void showDBStatus();

  void setProgressMaximum(int value);
  int progressMaximum();
  void setProgressValue(int value);
  int progressValue();
  void increaseProgressValue(int value=1);
  void hideProgressBar();
  void showMessage(QString message);
private slots:
  void directoryMonitorDestroyed();
  void seedlinkDestroyed();
signals:
  void asyncSetProgressMaximum(int value);
  void asyncSetProgressValue(int value);
  void asyncIncreaseProgressValue(int value);
  void asyncShowMessage(QString message);
protected:
  virtual void dragEnterEvent(QDragEnterEvent * event);
  virtual void dragMoveEvent(QDragMoveEvent * event);
  virtual void dropEvent(QDropEvent * event);
  virtual void closeEvent(QCloseEvent * event);
  bool warnBeforeClear (QString action);

  SignalDatabase * _db;

  ToolFactory * _toolFactory;
  SeedLinkLoader * _seedLink;
  DirectoryMonitor * _directoryMonitor;
  SubPoolWindow * _subPoolWin;

  FileView * _fileList;
  GroupWidget * _groupList;
  SeismicEventWidget * _eventList;
  WaveformConsole * _waveformConsole;
  LogView * _log;

  QAction * _fileExportAction;
  QAction * _fileImportAction;

  QAction * _viewTableAction;
  QAction * _viewGraphicAction;
  QAction * _viewMapAction;
  QAction * _viewChronogramAction;

  QAction * _viewSetDataFieldsAction;

  QAction * _fftTimeAction;
  QAction * _fftAmplitudeAction;
  QAction * _fftPhaseAction;

  QMenu * _importSignalMenu;
  QMenu * _editMenu;
  QMenu * _viewMenu;
  QMenu * _toolsMenu;
  QMenu * _waveMenu;
  QMenu * _windowsMenu;
  QMenu * _toolBarsMenu;

  QToolBar * _toolsBar;
  ProgressBar * _progressBar;
  QTimer _progressHideTimer;

  bool _tableEditionLocked;

  uint _dbStatusChecksum;
  QTimer _dbStatusTimer;
private:
  void addActions();
  void addFileActions();
  void addViewActions();
  void addEditActions();
  void addWaveformActions();
  void addToolsActions();
  void addSplitGroup(SubSignalPool& splitGroupSubPool, AbstractSignalGroup * parent,
                     ExpressionString& gName, SignalExpressionContext& context,
                     int viewerIndex, int groupIndex);
};

#endif
