/***************************************************************************
**
**  This file is part of dinverdc.
**
**  dinverdc is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  dinverdc is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-03-02
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCompatibility.h>
#include <QGpCoreTools.h>
#include "EllipticityPeakTargetWidget.h"

/*!
  \class EllipticityPeakTargetWidget EllipticityPeakTargetWidget.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
EllipticityPeakTargetWidget::EllipticityPeakTargetWidget(QWidget * parent)
    : QWidget(parent)
{
  TRACE;
  setupUi(this);

  connect(f0Edit, SIGNAL(valueChanged(double)), this, SIGNAL(peakModified()));
  connect(devF0Edit, SIGNAL(valueChanged(double)), this, SIGNAL(peakModified()));
}

/*!
  Description of destructor still missing
*/
EllipticityPeakTargetWidget::~EllipticityPeakTargetWidget()
{
  TRACE;
  WidgetRect::saveRect(this);
}

void EllipticityPeakTargetWidget::resizeEvent(QResizeEvent * e)
{
  TRACE;
  WidgetRect::record(this);
  QWidget::resizeEvent(e);
}

void EllipticityPeakTargetWidget::setValue(const RealStatisticalValue& val)
{
  TRACE;
  f0Edit->setValue(val.mean());
  devF0Edit->setValue(val.stddev());
}

RealStatisticalValue EllipticityPeakTargetWidget::value() const
{
  TRACE;
  RealStatisticalValue val(f0Edit->value());
  val.setStddev(devF0Edit->value());
  return val;
}

void EllipticityPeakTargetWidget::setEditable(bool e)
{
  TRACE;
  f0Edit->setEnabled(e);
  devF0Edit->setEnabled(e);
  loadBut->setEnabled(e);
}

void EllipticityPeakTargetWidget::on_loadBut_clicked()
{
  TRACE;
  QString fileName=Message::getOpenFileName(tr("Load peak to fit"),
                                                  tr("Geopsy HV output(*.hv);;"
                                                     "na_viewer report (*.report)" ));
  if( !fileName.isEmpty()) {
    if(CompatInversionReport::isRightFormat(fileName) )
      loadReport(fileName);
    else {
      QFile f(fileName);
      if(f.open(QIODevice::ReadOnly) ) {
        QTextStream s(&f);
        QString str=s.readLine().trimmed();
        f.close();
        if(str.startsWith( "# GEOPSY output version" ))
          loadHVGeopsy(fileName);
        else {
          Message::warning(MSG_ID, tr("Loading a H/V peak"), tr("Unknown format for file %1").arg(fileName),
                                Message::ok());
        }
      } else {
          Message::warning(MSG_ID, tr("Loading a H/V peak"), tr("Cannot open file %1").arg(fileName),
                                Message::ok());
      }
    }
    emit peakModified();
  }
}

void EllipticityPeakTargetWidget::loadReport(QString fileName)
{
  TRACE;
  CompatInversionReport report(false, fileName);
  if(report.currentEllipticity()==0) {
    Message::warning(MSG_ID, tr("Importing report"),
                         tr("Report %1 contains no ellipticity curve\n").arg(fileName), Message::ok(), true);
    return ;
  }
  CompatEllipticityData * ell=report.loadEllipticityGoal();
  f0Edit->setValue(ell->f0());
  devF0Edit->setValue(ell->devf0());
  delete ell;
}

void EllipticityPeakTargetWidget::loadHVGeopsy(QString fileName)
{
  TRACE;
  // open values as 2 or 3 columns
  QFile f(fileName);
  if( !f.open(QIODevice::ReadOnly) ) {
    Message::warning(MSG_ID, tr("Loading Geopsy HV output ..."),
                         tr("Impossible to access to file %1").arg(fileName), Message::cancel(), true);
    return ;
  }
  QTextStream s(&f);
  while( !s.atEnd()) {
    QString str=s.readLine().trimmed().simplified();
    if(str.left(17)=="# f0 from windows" ) {
      double f0=str.section(QRegExp( "[ \t]" ), 4, 4).toDouble();
      f0Edit->setValue(f0);
      devF0Edit->setValue(f0-str.section(QRegExp( "[ \t]" ), 5, 5).toDouble());
    }
  }
}
