/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-07-17
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "ActivityMonitor.h"
#include "LineLayer.h"
#include "PlotLine2D.h"

namespace SciFigs {

/*!
  \class ActivityMonitor ActivityMonitor.h
  \brief Brief description of class still missing

  Count number of hits per seconds and plot the variations of this number with time
*/

/*!
  Description of constructor still missing
*/
ActivityMonitor::ActivityMonitor(QWidget *parent, Qt::WindowFlags f)
    : AxisWindow(parent, f)
{
  TRACE;
  xAxis()->setEnabled(false);
  yAxis()->setEnabled(false);
  xAxis()->setRange(0.0,100.0);
  yAxis()->setRange(0.0,10.0);
  graphContent()->setGridLines(true);
  xAxis()->setAutoTicks(false);
  yAxis()->setAutoTicks(false);
  xAxis()->setMajorTicks(10.0);
  xAxis()->setMinorTicks(2.0);
  yAxis()->setMajorTicks(0.2);
  yAxis()->setMinorTicks(0.2);
  _layer=new LineLayer(this);
  _layer->setReferenceLine(new PlotLine2D);
  _layer->addLine(Pen( Qt::green, 0.75), Symbol());
  _measurementTimer.setInterval(1000);
  connect(&_measurementTimer, SIGNAL(timeout()), this, SLOT(measure()));
  _measurementTimer.start();
  _hits=0;
  updateExternalGeometry();
}

/*!
  Description of destructor still missing
*/
ActivityMonitor::~ActivityMonitor()
{
  TRACE;
}

void ActivityMonitor::measure()
{
  TRACE;
  static const QDateTime reference=QDateTime::currentDateTime().addYears(-1);
  Curve<Point2D>& curve=static_cast<PlotLine2D *>(_layer->line(0))->curve();
  int t=reference.secsTo(QDateTime::currentDateTime());
  curve.append(Point2D(t,_hits));
  _hits=0;
  // Adjust X axis
  double maxTime=xAxis()->maximum();
  double deltaTime=maxTime-xAxis()->minimum();
  if(t>maxTime) {
    maxTime=t;
    xAxis()->setRange(maxTime-deltaTime, maxTime);
  }
  // Remove first samples of the curve
  double min=xAxis()->minimum();
  while(!curve.isEmpty() && curve.first().x()<min) {
    curve.remove(0);
  }
  double maxY=curve.at(curve.maximumY(curve.count()-10)).y();
  if(maxY==0.0) maxY=1.0;
  yAxis()->setRange(0.0,maxY);
  yAxis()->setMajorTicks(0.2*maxY);
  yAxis()->setMinorTicks(0.2*maxY);
  deepUpdate();
}

/*!
  By default, it is measuring the number of hits per seconds.
*/
void ActivityMonitor::setMeasurementDelay(double d)
{
  TRACE;
  _measurementTimer.setInterval((int)round(d*1000.0));
}

/*!
  Set display range in seconds. Default is 120 seconds.
*/
void ActivityMonitor::setDisplayRange(double t)
{
  TRACE;
  double maxTime=xAxis()->maximum();
  xAxis()->setRange(maxTime-t, maxTime);
  xAxis()->setMajorTicks(t*0.1);
  xAxis()->setMinorTicks(t*0.02);
  deepUpdate();
}

/*!
  Avoid properties general widget
*/
void ActivityMonitor::mousePressEvent(QMouseEvent * )
{

}

/*!
  Avoid properties general widget
*/
void ActivityMonitor::mouseReleaseEvent(QMouseEvent * )
{

}

/*!
  Avoid properties general widget
*/
void ActivityMonitor::mouseDoubleClickEvent(QMouseEvent * )
{
}

} // namespace SciFigs
