/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2005-12-21
**  Copyright: 2005-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef PROFILE_H
#define PROFILE_H

#include <QGpCoreMath.h>

#include "QGpCoreWaveDLLExport.h"

namespace QGpCoreWave {

class QGPCOREWAVE_EXPORT Profile
{
public:
  inline void operator=(const Profile& p);
  inline void setDepth(int i, double depth);
  inline void setValue(int i, double value);
  void addValue(double depth, double value);
  inline void clear();
  inline void resize(int n);
  inline void collectDepths(QVector<double>& depths) const;
  inline void writeReport(QDataStream& s) const;
  inline void readReport(QDataStream& s);
  void resample(const Profile& p, const QVector<double>& baseD);
  void resample(const QVector<double>& baseD);
  inline void resample(const QVector<double>& baseD1, const QVector<double>& baseD2);
  void setSamples(const QVector<double>& baseD);
  const QVector<double>& depths() const {return _depths;}
  const QVector<double>& values() const {return _values;}
  QVector<double>& values() {return _values;}
  static void report2plot(QDataStream& s, Point2D * points);
  double interpoleAt(double depth);
  double uniformAt(double depth);
  void average();
  void inverse();
  int count() const {return _depths.count();}
  void toStream(QTextStream& s, bool plotMode) const;
  Curve<Point2D> curve() const;
  void print() const;
private:
  QVector<double> _depths;
  QVector<double> _values;
};

inline void Profile::setDepth(int i, double depth)
{
  TRACE;
  _depths[i]=depth;
}

inline void Profile::setValue(int i, double value)
{
  TRACE;
  _values[i]=value;
}

inline void Profile::collectDepths(QVector<double>& depths) const
{
  TRACE;
  depths << _depths;
}

inline void Profile::clear()
{
  TRACE;
  _depths.clear();
  _values.clear();
}

inline void Profile::resize(int n)
{
  TRACE;
  _depths.resize(n);
  _values.resize(n);
}

inline void Profile::writeReport(QDataStream& s) const
{
  TRACE;
  int n=_depths.count();
  s << n;
  for(int i=0;i<n;i++) {
    s << _depths.at(i);
    s << _values.at(i);
  }
}

inline void Profile::readReport(QDataStream& s)
{
  TRACE;
  int n;
  s >> n;
  if(n<0 || n>10000) {
    // Probably a corrupted file
    return;
  }
  resize(n);
  double d,v;
  for(int i=0;i<n;i++) {
    s >> d >> v;
    setDepth(i, d);
    setValue(i, v);
  }
}

inline void Profile::operator=(const Profile& p)
{
  TRACE;
  _depths=p._depths;
  _values=p._values;
}

inline void Profile::resample(const QVector<double>& baseD1, const QVector<double>& baseD2)
{
  TRACE;
  resample(baseD1);
  resample(baseD2);
}

} // namespace QGpCoreWave

#endif // PROFILE_H
